<?php declare(strict_types=1);

namespace Drupal\affiliate\Entity;

use Drupal;
use Drupal\Core\Entity\EntityDefinitionUpdateManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\systemx\AffiliateNameItem;

class EntityDefinitionUpdateHelper
{
  public function __construct(private readonly EntityDefinitionUpdateManagerInterface $definitionUpdateManager) {}

  public function installDefinitions(&$sandbox = []): bool
  {

    $entity_type = $this->definitionUpdateManager->getEntityType('user');
    $field_storage_definitions = $this->getStorageDefinitions(true);

    $this->definitionUpdateManager->updateFieldableEntityType($entity_type, $field_storage_definitions, $sandbox);
    Drupal::getContainer()->get('entity_type.manager')?->clearCachedDefinitions();
    Drupal::getContainer()->get('entity_type.bundle.info')?->clearCachedBundles();
    Drupal::getContainer()->get('entity_field.manager')?->clearCachedFieldDefinitions();

    /** @var \Drupal\Core\Entity\EntityFieldManager $entityFieldManager */
    $entityFieldManager = \Drupal::service('entity_field.manager');

    return !$this->definitionUpdateManager->needsUpdates();
  }

  public function uninstallDefinitions(): bool
  {
    $entity_type = $this->definitionUpdateManager->getEntityType('user');
    $result = true;
    if(null === $entity_type) {
      return false;
    }

    foreach($this->getStorageDefinitions() as $storageDefinition) {
      $result = $this->definitionUpdateManager->uninstallFieldStorageDefinition($storageDefinition);
    }

    return $result;
  }

  /**
   * @return \Drupal\Core\Field\FieldStorageDefinitionInterface[]
   */
  public function getStorageDefinitions(bool $includeExistingDrupal = false): array
  {
    $field_storage_definitions = [];

    if($includeExistingDrupal) {
      /** @var \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface $last_installed_schema_repository */
      $last_installed_schema_repository = \Drupal::service('entity.last_installed_schema.repository');
      $field_storage_definitions = $last_installed_schema_repository->getLastInstalledFieldStorageDefinitions('user');
    }

    $field_storage_definitions['field_affiliate_display_name'] = $this->getBaseFieldDefinitionDefaults(
      required: true,
      type: 'string',
      name: 'field_affiliate_display_name',
      label: 'Display Name',
      description: 'Full Display Name',
      settings: [
        'max_length' => 150,
        'min_length' => 8,
      ]
    );

    //$field_storage_definitions['field_affiliate_display_name']->setClass(AffiliateNameItem::class);

    $field_storage_definitions['field_affiliate_title'] = $this->getBaseFieldDefinitionDefaults(
      required: false,
      type: 'string',
      name: 'field_affiliate_title',
      label: 'Professional Title',
      description: 'Professional work Title',
      settings: [
        'max_length' => 150,
      ]
    );

    $field_storage_definitions['field_affiliate_company'] = $this->getBaseFieldDefinitionDefaults(
      required: true,
      type: 'string',
      name: 'field_affiliate_company',
      label: 'Affiliate Company',
      description: 'The member company you work for.',
      settings: [
        'max_length' => 255,
      ]
    );

    return $field_storage_definitions;
  }

  public function getBaseFieldDefinitionDefaults(
    bool $required,
    string $type,
    string $name,
    string $label,
    string $description,
    array $settings): BaseFieldDefinition
  {
    return BaseFieldDefinition::create($type)
      ->setTargetEntityTypeId('user')
      ->setTargetBundle(NULL)
      ->setProvider('user')
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'hidden',
        'weight' => -15
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDefaultValue('')
      ->setName($name)
      ->setLabel(new TranslatableMarkup($label))
      ->setDescription(new TranslatableMarkup($description))
      ->setSettings($settings)
      ->setRequired($required)
      ;
  }
}
