<?php

namespace Drupal\affiliate\Form;

use Drupal;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Drupal\user\Entity\Role;
use EasyRdf\Literal\HTML;
use EasyRdf\Utils;
use function current;
use function dump;
use function filter_default_format;
use function uasort;
use function user_roles;

/**
 * Configure Affiliate settings for this site.
 */
class SettingsForm extends ConfigFormBase {


  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'affiliate_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['affiliate.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $fieldMap = [];
    $fields = Drupal\field\Entity\FieldConfig::loadMultiple();

    /** @var \Drupal\user\PermissionHandler $permissionHandler */
    $permissionHandler = Drupal::service('user.permissions');
    $permissions = [];

    $affiliateConfig = $this->config('affiliate.settings');
    $lockedFields = $affiliateConfig->get('locked_fields');

    foreach($fields as $field) {

      if('node' !== $field->getTargetEntityTypeId()) {
        continue;
      }

      $bundleName = Drupal\node\Entity\NodeType::load($field->getTargetBundle())?->get('name') ?? $field->bundle();

      $fieldMap[$field->getName()] = [
         'title' => sprintf('%s (%s)', $field->getLabel(), $field->getType()),
         'bundle_name' => $bundleName,
        ];
    }


    foreach($permissionHandler->getPermissions() as $permission => $definition)
    {
      $title = (new HTML((string)$definition['title']))->stripTags();
      $permissions[$permission] = $title;
    }

    uasort($permissions, static fn($permission1, $permission2) => $permission1 <=> $permission2);


    $node = null;
    if(null !== $nodeId = $affiliateConfig->get('contact_page')) {
      $node = Node::load($nodeId);
    }

    $form['contact_page'] = [
      '#title' => $this->t('Default contact page'),
      '#description' => $this->t('The page that members should be directed to for contacting an administrator.'),
      '#type' => 'entity_autocomplete',
      '#target_type' => 'node',
      '#default_value' => $node,
    ];

    $form['login_link_text'] = [
      '#title' => $this->t('Login link text'),
      '#description' => $this->t('Link text that logged out users see on locked nodes.'),
      '#type' => 'textfield',
      '#default_value' => $affiliateConfig->get('login_link_text') ?? $this->t('Affiliates: login to view related content.'),
    ];

    if(null === $affiliateConfig->get('affiliate_permission')) {
      $affiliateConfig->set('affiliate_permission', 'view affiliate affiliate content');
      $affiliateConfig->save();
    }

    $form['affiliate_permission'] = [
      '#type' => 'select',
      '#options' => $permissions,
      '#multiple' => FALSE,
      '#title' => $this->t('Affiliate Permission'),
      '#description' => $this->t('Drupal permission to give access to content marked member only.'),
      '#default_value' => $affiliateConfig->get('affiliate_permission'),
    ];

    $form['locked_fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Fields that should be restricted to affiliates only.'),
    ];

    foreach($fieldMap as $fieldKey => $field) {
      $form['locked_fields'][$fieldKey] = [
        '#type' => 'checkbox',
        '#value' => $fieldKey,
        '#title' => $field['title'],
        '#description' => $this->t('Used in @bundleName', ['@bundleName' => $field['bundle_name']]),
        '#checked' => $lockedFields[$fieldKey] ?? false,
      ];
    }
    $form['landing_page'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Landing page markup.'),
      '#description' => $this->t('The markup that will appear on the members login page.'),
      '#default_value' => $affiliateConfig->get('landing_page')['value'] ?? null,
      '#format' => $affiliateConfig->get('landing_page')['format'] ?? filter_default_format(),
      '#rows' => 5,
      '#resizeable' => 'vertical',
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {

    dump($form_state->getValue('affiliate_role'), 'affiliate_role');

    $this->config('affiliate.settings')
      ->set('affiliate_role', $form_state->getValue('affiliate_role'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
