<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\CAP\Authorization;

use JetBrains\PhpStorm\ArrayShape;
use JetBrains\PhpStorm\Pure;

/**
 * Class TokenStorage
 *
 * @package Suee\CAP\Authorization
 */
class Token
{
    private array $tokenData;
    private string $accessToken = '';
    private string $tokenType = '';
    private int $expiresIn = 0;
    private int $expiresAt = 0;
    private string $scope = '';
    private string $jti = '';
    private array $errors = [];
    private bool $hasErrors = false;

    public function __construct(array $tokenData)
    {
        $this->tokenData = $tokenData;
        $this->setErrors();
        if (!$this->hasErrors()) {
            $this
                ->setAccessToken($tokenData['access_token'])
                ->setTokenType($this->tokenData['token_type'])
                ->setExpiresIn($this->tokenData['expires_in'])
                ->setExpiresAt(time() + $this->tokenData['expires_in'])
                ->setScope($this->tokenData['scope'])
                ->setJti($this->tokenData['jti']);
        }
    }

    #[Pure]
    #[ArrayShape([
      'access_token' => "string",
      'token_type'   => "string",
      'expires_in'   => "int",
      'expires_at'   => "int",
      'scope'        => "string",
      'jti'          => "string",
    ])]
    public function __serialize(): array
    {
        return [
          'access_token' => $this->getAccessToken(),
            'token_type' => $this->getTokenType(),
            'expires_in' => $this->getExpiresIn(),
            'expires_at' => $this->getExpiresAt(),
            'scope' => $this->getScope(),
            'jti' => $this->getJti(),
        ];
    }

    public function __unserialize(array $data): void
    {
        $this
          ->setAccessToken($data['access_token'])
          ->setTokenType($data['token_type'])
          ->setExpiresIn($data['expires_in'])
          ->setExpiresAt($data['expires_at'])
          ->setScope($data['scope'])
          ->setJti($data['jti']);
    }

    /**
     * @return array
     */
    #[ArrayShape(['Authorization' => "string"])]
    public function getTokenHeader(): array
    {
        return ['Authorization' => sprintf('Bearer %s', $this->accessToken)];
    }

    /**
     * @return int
     */
    public function getExpiresAt(): int
    {
        return $this->expiresAt;
    }

    public function hasErrors(): bool
    {
        return $this->hasErrors;
    }

    /**
     * @return \Suee\CAP\Authorization\TokenError
     */
    public function getErrors(): TokenError
    {
        return new TokenError($this->errors);
    }

    /**
     * @return array
     */
    public function getTokenData(): array
    {
        return $this->tokenData;
    }

    /**
     * @return string
     */
    public function getAccessToken(): string
    {
        return $this->accessToken;
    }

    /**
     * @return string
     */
    public function getTokenType(): string
    {
        return $this->tokenType;
    }

    /**
     * @return int
     */
    public function getExpiresIn(): int
    {
        return $this->expiresIn;
    }

    /**
     * @return string
     */
    public function getScope(): string
    {
        return $this->scope;
    }

    /**
     * @return string
     */
    public function getJti(): string
    {
        return $this->jti;
    }

    /**
     * @return \Suee\CAP\Authorization\Token
     */
    public function setErrors(): static
    {
        if (isset($this->tokenData['error'])) {
            $this->errors = $this->tokenData;
            $this->hasErrors = true;
        } else {
            $this->hasErrors = false;
        }

        return $this;
    }

    private function setExpiresAt(int $expiresAt): static
    {
        $this->expiresAt = $expiresAt;
        return $this;
    }

    /**
     * @param string $accessToken
     *
     * @return $this
     */
    private function setAccessToken(string $accessToken): static
    {
        $this->accessToken = $accessToken;
        return $this;
    }

    /**
     * @param $tokenType
     *
     * @return $this
     */
    private function setTokenType(string $tokenType): static
    {
        $this->tokenType = $tokenType;
        return $this;
    }

    /**
     * @param int $expiresIn
     *
     * @return $this
     */
    private function setExpiresIn(int $expiresIn): static
    {
        $this->expiresIn = $expiresIn;
        return $this;
    }

    /**
     * @param string $scope
     *
     * @return $this
     */
    private function setScope(string $scope): static
    {
        $this->scope = $scope;
        return $this;
    }

    /**
     * @param string $jti
     *
     * @return $this
     */
    private function setJti(string $jti): static
    {
        $this->jti = $jti;

        return $this;
    }
}
