<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\CAP\Authorization;

use GuzzleHttp\Client;
use Stash\Driver\FileSystem;
use Stash\Pool;
use Webmozart\Json\JsonDecoder;
use Webmozart\Json\ValidationFailedException;

/**
 * Class Token
 *
 * @package Suee\CAP\Authorization
 */
class TokenFactory
{
    /**
     *
     */
    public const DEFAULT_TOKEN_URL = 'https://authz.stanford.edu/oauth/token';

    /**
     * @var string
     */
    private string $username;
    /**
     * @var string
     */
    private string $password;
    /**
     * @var Client
     */
    private Client $client;
    /**
     * @var int
     */
    private int $expiresIn;
    /**
     * @var int
     */
    private int $expires;
    /**
     * @var array
     */
    private array $options;

    /**
     * Token constructor.
     *
     * @param       $username
     * @param       $password
     * @param array $options
     */
    public function __construct(string $username, string $password, array $options = [])
    {
        $this->username = $username;
        $this->password = $password;
        $this->options = $options;
    }

    /**
     * @return \Suee\CAP\Authorization\Token|null
     */
    public function request(): ?Token
    {
        $currentTime = time();
        $jsonDecoder = new JsonDecoder();
        $jsonDecoder->setObjectDecoding(JsonDecoder::ASSOC_ARRAY);
        $item = $this->getCacheDriver()->getItem('tokenStorage');

        if ($item->isMiss()) {
            $response = $this->requestNewToken();
            try {
                $jsonArray = $jsonDecoder->decode($response);
            } catch (ValidationFailedException $e) {
                error_log(sprintf('There was an error decoding the returned json on line %s of %s.', $e->getLine(), $e->getFile()), 0);
                return null;
            }
            $token = new Token($jsonArray);
        } else {
            $token = $item->get();
        }

        if ($item->isMiss()) {
            $item->clear();
            $item->set($token);
            $item->expiresAfter($token->getExpiresIn());
            $item->save();
        }

        return $token;
    }

    /**
     * @return \Stash\Pool
     */
    private function getCacheDriver(): Pool
    {
        $options = ['path' => sprintf('%s/cap/', sys_get_temp_dir())];
        $driver = new FileSystem($options);
        return new Pool($driver);
    }

    /**
     * @param $duration
     *
     * @return $this
     */
    private function setExpires($duration): static
    {
        $dt = new \DateTimeImmutable();
        $dt->setTimestamp($dt->getTimestamp() + $duration);

        $this->expires = $dt->getTimestamp();
        return $this;
    }

    private function requestNewToken(): string
    {
        return $this->getClient()->request('POST')->getBody()->getContents();
    }

    /**
     * @return \GuzzleHttp\Client
     */
    private function getClient()
    {
        $options = [
            'base_uri' => $this->options['base_uri'] ?? self::DEFAULT_TOKEN_URL,
            'http_errors' => false,
            'query' => ['grant_type' => 'client_credentials'],
            'auth' => [$this->username, $this->password],
        ];

        return new Client($options);
    }
}
