<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\CAP\Request;

use ArrayObject;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

use Psr\Http\Message\ResponseInterface;
use Suee\CAP\Authorization\Token;

abstract class AbstractRequest
{
    /**
     * @var string|null
     */
    protected ?string $capApiEndpoint = null;

    /**
     * @var string
     */
    protected string $capApiUrl = 'https://api.stanford.edu';

    /**
     * @var string
     */
    protected string $capApiVersion = 'v1';

    /**
     * @var \Suee\CAP\Authorization\Token
     */
    private Token $token;

    private ArrayObject $params;
    private ArrayObject $queryParams;

    private Client $client;

    /**
     * Request constructor.
     *
     * @param \Suee\CAP\Authorization\Token$token
     */
    public function __construct(Token $token)
    {
        $this->token = $token;
        $this->client = new Client(['base_uri' => $this->capApiUrl, 'http_errors' => false]);
        $this->params = new ArrayObject();
        $this->queryParams = new ArrayObject();
    }

    /**
     * @param $key
     * @param $value
     *
     * @return \Suee\CAP\Request\AbstractRequest
     */
    public function setQueryParam($key, $value): static
    {
        $this->queryParams->offsetSet($key, $value);

        return $this;
    }

    /**
     * @param $key
     *
     * @return mixed
     */
    public function getQueryParam($key): mixed
    {
        if ($this->queryParams->offsetExists($key)) {
            return $this->queryParams->offsetGet($key);
        }
        return null;
    }

    /**
     * @param array $params
     *
     * @return \Suee\CAP\Request\AbstractRequest
     */
    public function setParams(array $params): static
    {
        $this->params = new ArrayObject($params);

        return $this;
    }

    /**
     * @param $key
     * @param $value
     *
     * @return \Suee\CAP\Request\AbstractRequest
     */
    public function setParam($key, $value): static
    {
        $this->params->offsetSet($key, $value);

        return $this;
    }

    /**
     * @param $key
     *
     * @return mixed
     */
    public function getParam($key): mixed
    {
        if ($this->params->offsetExists($key)) {
            return $this->params->offsetGet($key);
        }
        return null;
    }

    /**
     * @return \Psr\Http\Message\ResponseInterface|null
     */
    public function getRequest(): ?ResponseInterface
    {
        $authHeader = $this->token->getTokenHeader();
        try {
            return $this->client->get(sprintf(
                '/%s/%s',
                $this->capApiEndpoint,
                $this->capApiVersion
            ), [
            'headers' => $this->token->getTokenHeader(),
            'query'   => $this->queryParams->getArrayCopy(),
          ])->withAddedHeader('Authorization', $authHeader['Authorization']);
        } catch (GuzzleException) {
            return null;
        }
    }
}
