<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\CAP\Image;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Pool;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Suee\CAP\Authorization\Token;
use Webmozart\Json\JsonDecoder;
use Webmozart\Json\ValidationFailedException;

/**
 * Class CapProfileImageGenerator
 *
 * @author R. Eric Wheeler <reric@ee.stanford.edu>
 *
 * Takes an array of people that have https://profiles.stanford.edu profiles
 * and retrieves a profile image if it exists.
 */
class CapProfileImageGenerator
{
    public const SUNET_QUERY = 'uids';
    public const NAME_QUERY = 'name';

    /**
     * @var array
     */
    private $images;

    /**
     * @var array
     */
    private $person;

    /**
     * @var string
     */
    private $apiBase;

    /**
     * @var \Suee\CAP\Authorization\Token
     */
    private $token;

    /**
     * @var \GuzzleHttp\Client
     */
    private $client;

    /**
     * @var bool
     */
    private $onlyUpdateMissing;

    /**
     * @var string
     */
    private $nameQueryType = 'psName';

    /**
     * @var string
     */
    private $defaultBlankImage = 'https://profiles.stanford.edu/images/profile-large-blank.png';

    /**
     * @var array
     */
    private $hasNoImageCollection = [];

    /**
     * CapProfileImageGenerator constructor.
     *
     * @param \Suee\CAP\Authorization\Token $token
     * @param string                                         $apiBase
     */
    public function __construct(Token $token, $apiBase = 'https://api.stanford.edu/profiles/v1/')
    {
        $this->apiBase = $apiBase;
        $this->images = [];
        $this->token = $token;
        $guzzleConfig = [
      'base_uri' => $this->apiBase,
      'headers' => $this->token->getTokenHeader(),
    ];
        $this->client = new Client($guzzleConfig);
    }

    /**
     * @param $defaultBlankImage
     */
    public function setDefaultBlankImage($defaultBlankImage): void
    {
        $this->defaultBlankImage = $defaultBlankImage;
    }

    /**
     * @return string
     */
    public function getDefaultBlankImage()
    {
        return $this->defaultBlankImage;
    }

    /**
     * @return array
     */
    public function getData()
    {
        return $this->images;
    }

    /**
     * @param array $people
     *
     * @return \Generator
     */
    public function generator(array $people)
    {
        $totalPeople = count($people);
        $currentCount = 1;
        foreach ($people as $person) {
            $this->person = $person;

            variable_set('ee_orglist_image_process_progress', ['name' => $this->person['displayName'], 'current' => $currentCount, 'total' => $totalPeople]);

            if (array_key_exists('sunetId', $this->person) && !empty($this->person['sunetId'])) {
                yield new Request('GET', sprintf('?%s=%s', self::SUNET_QUERY, $this->person['sunetId']));
            }

            $currentCount++;
        }
    }

    /**
     * @param bool $onlyUpdateMissing
     */
    public function setOnlyUpdateMissing($onlyUpdateMissing = true): void
    {
        $this->onlyUpdateMissing = $onlyUpdateMissing;
    }

    /**
     * @return bool
     */
    public function isOnlyUpdateMissing()
    {
        return $this->onlyUpdateMissing;
    }
    /**
     * @param $people
     *
     * @return array
     */
    public function generate($people)
    {
        $pool = new Pool($this->client, $this->generator($people), [
      'concurrency' => 1,
      'fulfilled' => function (Response $response, $index): void {
          $image = null;
          $contents = $response->getBody()->getContents();
          //file_put_contents(sprintf('%s/%s.json', __DIR__, $this->person['sunetId']), $contents);

          $image = $this->getImageData($contents);

          $this->images[$this->person['id']] = $image;
          watchdog(
              'ee_orglist',
              t(
                 'Retrieved image (@image) for @displayName, with id @id',
                 [
                 '@image'       => $image,
                 '@displayName' => $this->person['displayName'],
                 '@id' => $this->person['id'],
               ]
             ),
              [],
              WATCHDOG_INFO
          );
      },
      'rejected' => function (ClientException $reason, $index): void {
          $this->images[$this->person['id']] = $this->getDefaultBlankImage();
          $this->hasNoImageCollection[$this->person['id']] = $this->person;
          watchdog(
              'ee_orglist',
              t(
              'Failed to get image for @displayName. Message : @message, Code: @code',
              [
            '@displayName' => $this->person['displayName'],
            '@message' => $reason->getMessage(),
            '@code' => $reason->getCode(),
          ]
          )
          );
      },
    ]);
        $promise = $pool->promise();
        $promise->wait();
        variable_set('ee_orglist_has_no_image', $this->hasNoImageCollection);
        return $this->images;
    }

    /**
     * @param $response
     *
     * @return string|null
     */
    private function getImageData($response)
    {
        $decoder = new JsonDecoder();
        try {
            $data = $decoder->decode($response);
        } catch (ValidationFailedException $e) {
            return $this->getDefaultBlankImage();
        }

        if ($data->count === 0 || (isset($data->values[0]->profilePhotos->big) && true === $data->values[0]->profilePhotos->big->placeholder)) {
            return $this->getDefaultBlankImage();
        }

        return $data->values[0]->profilePhotos->big->url ?? $this->getDefaultBlankImage();
    }
}
