<?php

namespace Drupal\ee_extra\Plugin\Field\FieldFormatter;

use Drupal;
use Drupal\Component\Utility\Html;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\HtmlTag;
use Drupal\Core\Render\HtmlResponse;
use Drupal\datetime\Plugin\Field\FieldFormatter\DateTimeCustomFormatter;
use Drupal\datetime\Plugin\Field\FieldFormatter\DateTimeFormatterBase;
use Drupal\datetime\Plugin\Field\FieldFormatter\DateTimePlainFormatter;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItem;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use function array_combine;
use function dump;

/**
 * Plugin implementation of the 'Stacked Date' formatter.
 *
 * @FieldFormatter(
 *   id = "ee_extra_stacked_date",
 *   label = @Translation("Stacked Date"),
 *   field_types = {
 *     "datetime",
 *     "date"
 *   }
 * )
 */
class StackedDateFormatter extends DateTimeFormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array
  {
    return [
        'style' => 'light',
        'show_full_month' => false,
        'show_day' => false,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array
  {
    $form = parent::settingsForm($form, $form_state);

    $form['style'] = [
      '#type' => 'select',
      '#title' => $this->t('Style'),
      '#options' => [
        'dark' => $this->t('Dark'),
        'light' => $this->t('Light'),
      ],
      '#default_value' => $this->getSetting('style'),
    ];

    $form['show_full_month'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show full month?'),
      '#default_value' => $this->getSetting('show_full_month'),
    ];

    $form['show_day'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show day?'),
      '#default_value' => $this->getSetting('show_day'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array
  {
    $summary = parent::settingsSummary();

    $summary[] = $this->t('Style: @style', ['@style' => ucfirst($this->getSetting('style'))]);
    $summary[] = $this->t('Show Full Month: @full_month', ['@full_month' => $this->getSetting('show_full_month') ? $this->t('Yes') : $this->t('No')]);
    $summary[] = $this->t('Show Day: @show_day', ['@show_day' => $this->getSetting('show_day') ? $this->t('Yes') : $this->t('No')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array
  {
    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = $this->buildDate($item->date);
    }

    $elements['#attached']['library'][] = 'ee_extra/ee_extra';

    return $elements;
  }

  public function buildDate(DrupalDateTime $date): array
  {
    $this->setTimeZone($date);

    return [
      '#theme' => 'ee_stacked_date',
      '#date_parts' => $this->dateFormat($date),
      '#style' => $this->getSetting('style'),
      '#show_day' => (bool)$this->getSetting('show_day'),
      '#cache' => [
        'contexts' => [
          'timezone',
        ],
      ],
    ];

  }

  /**
   * Leave this empty, we don't want to return a string.
   * Use dateFormat($date) instead.
   * @param $date
   *
   * @return string|void
   */
  protected function formatDate($date)
  {
  }

  /**
   * Creates a formatted date value as a string.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime $date
   *   A date object.
   *
   * @return array A formatted date string using the chosen format.
   *   A formatted date string using the chosen format.
   */
  protected function dateFormat(DrupalDateTime $date): array
  {
    $dateParts = [
      'month' => $this->getSetting('show_full_month') ? 'F' : 'M',
      'day' => '',
    ];

    if($this->getSetting('show_day')) {
      $dateParts['day'] = 'jS';
    }

    $timezone = $this->getSetting('timezone_override') ?: $date->getTimezone()->getName();

    $formattedArray = explode(
      ',',
      $this->dateFormatter->format(
        $date->getTimestamp(),
        'custom',
        sprintf('Y,%s,%s', ...array_values($dateParts)),
        $timezone !== '' ? $timezone : NULL)
    );

    return array_combine([
      'year',
      'month',
      'day',
    ], $formattedArray);

  }

  public static function isApplicable(FieldDefinitionInterface $field_definition): bool
  {
    return $field_definition->getFieldStorageDefinition()->getSetting('datetime_type') === DateTimeItem::DATETIME_TYPE_DATE;
  }
}
