<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu reric@ee.stanford.edu
 *
 * This Source Code Form is subject to the
 * terms of the Mozilla Public License, v. 2.0.
 *
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 *
 */

namespace Suee\Mupdf\Mutool;

use function count;
use Countable;
use function dirname;
use function file_exists;
use function is_dir;
use function is_writable;
use function sprintf;
use Suee\Mupdf\Mutool\Exception\InvalidMutoolsBinaryException;
use Symfony\Component\Process\ExecutableFinder;
use Symfony\Component\Process\Process;
use function sys_get_temp_dir;
use function trigger_error;
use function usort;

final class Merge implements Countable
{
    private string $mutoolBinary;
    private array $files = [];
    private string $workingDirectory;
    private array $options = [];

    public function __construct(string $mutoolBinary = '/usr/bin/mutool', $options = [])
    {
        if (!file_exists($mutoolBinary)) {
            $executableFinder = new ExecutableFinder();

            if (null === $mutool = $executableFinder->find('mutool')) {
                throw new InvalidMutoolsBinaryException(sprintf('The file "%s" does not exist.', $mutoolBinary));
            }

            trigger_error(sprintf('Could not find mutool at "%s", found at "%s".  Using "%s"', $mutoolBinary, $mutool, $mutool), E_NOTICE);

            $this->mutoolBinary = $mutool;
        } else {
            $this->mutoolBinary = $mutoolBinary;
        }

        foreach ($options as $option => $value) {
            if (in_array($option, MergeOptions::VALID_OPTIONS, true)) {
                $this->options[$option] = $value;
            } else {
                trigger_error(sprintf('Invalid option "%s", skipping.', $option), E_NOTICE);
            }
        }
    }

    public function setOption(string $option, ?string $value): Merge
    {
        if (in_array($option, MergeOptions::VALID_OPTIONS, true)) {
            $this->options[$option] = $value ?? true;
        }
        return $this;
    }

    public function getOption(string $option): ?string
    {
        return $this->options[$option] ?? null;
    }

    public function getOptions(): array
    {
        return $this->options;
    }

    public function setOptions(array $options): Merge
    {
        foreach ($options as $option => $value) {
            if (in_array($option, MergeOptions::VALID_OPTIONS, true)) {
                $this->options[$option] = $value;
            }
        }

        return $this;
    }

    public function addFile(string $file): Merge
    {
        if (!file_exists($file)) {
            trigger_error(sprintf('The file "%s" does not exist.  Skipping.', $file), E_WARNING);
            return $this;
        }

        $this->files[] = $file;

        return $this;
    }

    public function getFiles(): array
    {
        return $this->files;
    }

    public function usort(callable $callback): bool
    {
        return usort($this->files, $callback);
    }

    public function saveFile(string $outputDirectory, string $filename): bool
    {
        if (!$this->isDirectoryWritable($outputDirectory)) {
            trigger_error(sprintf('The directory "%s" is not writable. Can not write "%s".', dirname($filename), $filename), E_WARNING);
            return false;
        }
        $fullPath = sprintf('%s/%s', $outputDirectory, $filename);

        $command = [$this->mutoolBinary, 'merge', '-o', $fullPath];

        //if(!empty($this->options)) {

        //}

        foreach ($this->files as $file) {
            $command[] = $file;
        }

        $process = new Process($command);

        $process->setWorkingDirectory($this->getWorkingDirectory());

        return $process->run() === 0;
    }

    public function setWorkingDirectory(string $directory): Merge
    {
        if (!$this->isDirectoryWritable($directory)) {
            trigger_error(sprintf('"%s" is either not a directory or not writable.  Setting working directory to "%s".', $directory, sys_get_temp_dir()), E_USER_WARNING);
            $this->workingDirectory = sys_get_temp_dir();

            return $this;
        }

        $this->workingDirectory = $directory;

        return $this;
    }

    public function getWorkingDirectory(): string
    {
        return $this->workingDirectory ?? sys_get_temp_dir();
    }

    public function count(): int
    {
        return count($this->files);
    }

    private function isDirectoryWritable(string $directory): bool
    {
        return is_dir($directory) && is_writable($directory);
    }
}
