<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu reric@ee.stanford.edu
 *
 * This Source Code Form is subject to the
 * terms of the Mozilla Public License, v. 2.0.
 *
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at https://mozilla.org/MPL/2.0/.
 *
 *
 */

namespace Mutool;

use PHPUnit\Framework\TestCase;
use Suee\Mupdf\Mutool\Merge;

class MergeTest extends TestCase
{
    private const PDFS_DIRECTORY = __DIR__ . '/../pdfs';
    private const TEST_FILE = self::PDFS_DIRECTORY . '/cli-merged.pdf';
    private const TEST_FILES = [
      self::PDFS_DIRECTORY . '/merged.pdf',
      self::PDFS_DIRECTORY . '/merged2.pdf',
      self::PDFS_DIRECTORY . '/merged3.pdf',
      self::PDFS_DIRECTORY . '/merged4.pdf',
      self::PDFS_DIRECTORY . '/merged5.pdf',
      self::PDFS_DIRECTORY . '/merged6.pdf',
    ];

    /**
     * @covers \Suee\Mupdf\Mutool\Merge::count
     */
    public function testCount(): void
    {
        $mergeClass = new Merge();
        self::assertSame($mergeClass->count(), 0);
        $mergeClass->addFile(__DIR__ . '/../../composer.json');
        self::assertSame($mergeClass->count(), 1);
        $mergeClass->addFile(__DIR__. '/../../composer.lock');
        self::assertSame($mergeClass->count(), 2);
    }

    /**
     * @covers \Suee\Mupdf\Mutool\Merge::setWorkingDirectory
     * @covers \Suee\Mupdf\Mutool\Merge::getWorkingDirectory
     */
    public function testSetGetWorkingDirectory(): void
    {
        $mergeClass = new Merge();

        $this->expectWarning();
        $this->expectWarningMessage(
            sprintf(
                '"invalid_directory" is either not a directory or not writable.  Setting working directory to "%s".',
                sys_get_temp_dir()
            )
        );

        $mergeClass->setWorkingDirectory('invalid_directory');
        self::assertEquals(sys_get_temp_dir(), $mergeClass->getWorkingDirectory());

        $mergeClass->setWorkingDirectory(__DIR__);
        self::assertEquals(__DIR__, $mergeClass->getWorkingDirectory());
    }

    public function testSaveFile(): void
    {
        $mergeClass = new Merge();

        foreach (self::TEST_FILES as $testFile) {
            $mergeClass->addFile($testFile);
        }

        self::assertCount(6, $mergeClass->getFiles());

        $mergeClass->setWorkingDirectory(sys_get_temp_dir());
        $res = $mergeClass->saveFile(sys_get_temp_dir(), '/out.pdf');
        self::assertTrue($res);
        self::assertFileEquals(self::TEST_FILE, sys_get_temp_dir() . '/out.pdf');
    }

    public function testAddFile(): void
    {
        $mergeClass = new Merge();
        self::assertEmpty($mergeClass->getFiles());
        $mergeClass->addFile(self::PDFS_DIRECTORY . '/merged.pdf');
        self::assertNotEmpty($mergeClass->getFiles());
        self::assertSame([self::PDFS_DIRECTORY . '/merged.pdf'], $mergeClass->getFiles());
    }

    public function testUsort(): void
    {
        $mergeClass = new Merge();
        self::assertEmpty($mergeClass->getFiles());
        $mergeClass
            ->addFile(self::PDFS_DIRECTORY . '/merged2.pdf')
            ->addFile(self::PDFS_DIRECTORY . '/merged.pdf');

        self::assertSame([
          self::PDFS_DIRECTORY . '/merged2.pdf',
          self::PDFS_DIRECTORY . '/merged.pdf',
        ], $mergeClass->getFiles());
        $res = $mergeClass->usort(fn ($value, $value2) => $value <=> $value2);

        self::assertTrue($res);
        self::assertSame([
          self::PDFS_DIRECTORY . '/merged.pdf',
          self::PDFS_DIRECTORY . '/merged2.pdf',
        ], $mergeClass->getFiles());
    }

    public function testGetFiles(): void
    {
        $mergeClass = new Merge();
        $mergeClass->addFile(self::PDFS_DIRECTORY . '/merged.pdf')
            ->addFile(self::PDFS_DIRECTORY . '/merged2.pdf')
          ->addFile(self::PDFS_DIRECTORY . '/merged4.pdf');
        self::assertCount(3, $mergeClass->getFiles());
        self::assertSame([
          self::PDFS_DIRECTORY . '/merged.pdf',
          self::PDFS_DIRECTORY . '/merged2.pdf',
          self::PDFS_DIRECTORY . '/merged4.pdf',
        ], $mergeClass->getFiles());
        $mergeClass->addFile(self::PDFS_DIRECTORY . '/merged3.pdf');
        self::assertCount(4, $mergeClass->getFiles());
        self::assertSame([
          self::PDFS_DIRECTORY . '/merged.pdf',
          self::PDFS_DIRECTORY . '/merged2.pdf',
          self::PDFS_DIRECTORY . '/merged4.pdf',
          self::PDFS_DIRECTORY . '/merged3.pdf',
        ], $mergeClass->getFiles());
    }
}
