<?php

namespace Drupal\orglist\Commands;

use Consolidation\Log\ConsoleLogLevel;
use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\orglist\Cache\CacheUtility;
use Drupal\orglist\Request\Request;
use Drush\Commands\DrushCommands;
use Drush\Log\DrushLoggerManager;
use Symfony\Component\Console\Exception\InvalidArgumentException;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 *
 * See these files for an example of injecting Drupal services:
 *   - http://cgit.drupalcode.org/devel/tree/src/Commands/DevelCommands.php
 *   - http://cgit.drupalcode.org/devel/tree/drush.services.yml
 */
class OrglistCommands extends DrushCommands {

    private CacheUtility $cacheUtility;
    private Request $request;

    public function __construct()
    {
        parent::__construct();
        $this->cacheUtility = new CacheUtility();
        $this->request = new Request();
    }

  /**
   * Command description here.
   *
   *  param $arg1
   *   Argument description.
   * param array $options
   *   An associative array of options whose values come from cli, aliases, config, etc.
   * option option-name
   *   Description
   * @usage orglist:clear-cache (odc)
   *   Deletes all Orglist caches without refresh.  Use orglist:rebuild (orc) to delete and rebuild caches.
   *
   * @command orglist:clear-cache
   * @aliases occ
   */
  public function clearAllCache()
  {
      $this->logger()->notice(dt('Clearing all orglist caches ...'));
      $this->cacheUtility->deleteAll();
      $this->logger()->success(dt('Removed all orglist caches'));
     
  }

    /**
     * Deletes and rebuilds orglist caches.
     *
     * @usage orglist:rebuild-cache (orc)
     *   Deletes and rebuilds orglist caches.
     *
     * @command orglist:rebuild-cache
     * @aliases orc
     */
    public function rebuildAllCache()
    {
        $this->logger()->notice(dt('Rebuilding all orglist caches ...'));
        $result = $this->cacheUtility->refreshAll();
        if(\count($result) > 0) {
            $this->logger()->success(dt('Refreshed all orglist cache.'));
        } else {
            $this->logger()->error(dt('There was an error refreshing the cache.'));
        }
    }

    /**
     * Deletes and rebuilds orglist image caches.
     *
     * @usage orglist:image:clear-cache (oirc)
     *   Deletes image caches.
     *
     * @param string $type
     *  The type of image cache to delete (all, faculty, staff, courtesy, or consulting).
     *
     * @command orglist:image:clear-cache
     * @aliases oicc
     */
    public function clearImageCache(string $type = 'all'): void
    {
        $type = strtolower($type);
        if(!in_array($type, ['all', 'faculty', 'courtesy', 'consulting', 'staff'], true)) {
            $message = 'Available image cache types are (all, faculty, staff, courtesy, or consulting).';

            throw new InvalidArgumentException($message);
        }

        match($type) {
            default => $this->cacheUtility->deleteAllImageCache(),
            'faculty' => $this->cacheUtility->deleteImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_FACULTY),
            'courtesy' => $this->cacheUtility->deleteImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_COURTESY),
            'consulting' => $this->cacheUtility->deleteImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_CONSULTING),
            'staff' => $this->cacheUtility->deleteImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_STAFF),
        };

        $this->logger()->success(dt('Removed @type image cache.', ['@type' => $type]));
    }

    /**
     * Deletes and rebuilds orglist caches.
     *
     * @usage orglist:image:rebuild-cache (oirc)
     *   Deletes and rebuilds orglist image caches.
     *
     * @param string $type
     *   The type of image cache to rebuild (all, faculty, staff, courtesy, or consulting).
     *
     * @command orglist:image:rebuild-cache
     * @aliases oirc
     */
    public function rebuildAllImageCache(string $type = 'all')
    {
        $type = strtolower($type);
        if(!in_array($type, ['all', 'faculty', 'courtesy', 'consulting', 'staff', 'lecturer'], true)) {
            $message = 'Available image cache types are (all, faculty, staff, courtesy, lecturer, or consulting).';
            throw new InvalidArgumentException($message);
        }

        match($type) {
            default => $this->cacheUtility->rebuildAllImageCache(),
            'faculty' => $this->cacheUtility->rebuildImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_FACULTY),
            'courtesy' => $this->cacheUtility->rebuildImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_COURTESY),
            'consulting' => $this->cacheUtility->rebuildImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_CONSULTING),
            'staff' => $this->cacheUtility->rebuildImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_STAFF),
            'lecturer' => $this->cacheUtility->rebuildImageCacheOfType(CacheUtility::CACHE_KEY_IMAGE_LECTURER),
        };

        $this->logger()->success(dt('Removed and rebuilt @type image cache.', ['@type' => $type]));
    }
}
