<?php

namespace Drupal\orglist\Controller;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormBuilder;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Http\Exception\CacheableNotFoundHttpException;
use Drupal\Core\Pager\Pager;
use Drupal\Core\Pager\PagerManager;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Pager\PagerParameters;
use Drupal\Core\Render\Element\Form;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\orglist\Cache\CacheUtility;
use Drupal\orglist\Form\OrglistBaseSortForm;
use Drupal\orglist\Form\OrglistFacultySortForm;
use Drupal\orglist\Form\OrglistStaffSortForm;
use Drupal\orglist\Http\OrglistClientFactory;
use Drupal\orglist\Request\Request;
use Drupal\orglist\Security\Access;
use Drupal\orglist\Utils;
use Drupal\user\Entity\Role;
use Drupal\user\Plugin\views\field\Roles;
use GuzzleHttp\Client;
use JetBrains\PhpStorm\ArrayShape;
use Symfony\Component\HttpFoundation\Request as SymfonyRequest;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use function array_key_last;
use function array_pop;
use function count;
use function is_array;
use function uasort;

/**
 * Returns responses for Orglist routes.
 */
class OrglistController extends ControllerBase
{
    private SymfonyRequest $request;
    Private PagerManager $pagerManager;
    private CacheUtility $cacheUtility;
    private Access $accessChecker;

    public function __construct()
    {
        $this->request = \Drupal::request();
        $this->pagerManager = \Drupal::service('pager.manager');
        $this->cacheUtility = new CacheUtility();
        $this->accessChecker = new Access();

    }

    public function getFacultyTitle(): string
    {

        $areaName = null;

        if($this->request->query->has('areas')) {
            $areas = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_ORG_GROUP);
      
            foreach($areas as $area) {
                if($area['id'] === $this->request->query->get('areas')) {
                    $areaName = $area['name'];
                }
            }
        }

        return $this->t('Faculty @areaName', ['@areaName' => $areaName ? '- ' . $areaName : '']);
    }

    #[ArrayShape([
      "sort_form" => "array",
      "page" => "array",
      "pager" => "array"
    ])]
    public function faculty(): array
    {
        if(!$this->accessChecker->facultyIsEnabled()) {
            throw new NotFoundHttpException();
        }

        // Filter = Courtesy, Emeritus, Visiting
        // Default - Active

        $mainFacultyData = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_FACULTY);
        $courtesyFacultyData = [];

        if($this->accessChecker->courtesyIsEnabled()) {
            $courtesyFacultyData = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_COURTESY);
        }

        $facultyData = array_merge($mainFacultyData, $courtesyFacultyData);

        $imageData = $this->cacheUtility->getImageCache(CacheUtility::CACHE_KEY_IMAGE_FACULTY);

        $courtesyImageData = [];

        if($this->accessChecker->courtesyIsEnabled()) {
            $courtesyImageData = $this->cacheUtility->getImageCache(CacheUtility::CACHE_KEY_IMAGE_COURTESY);
        }

        $imageData = array_merge($imageData, $courtesyImageData);
        


        if(($typeId = $this->request->query->get('type')) && mb_strtolower($typeId) !== 'all' && mb_strtolower($typeId) !== 'active') {
            $personTypes = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_PERSON_TYPE);
            $personType = current(array_filter($personTypes, static function($people) use ($typeId) {
                if($people['id'] === $typeId) {
                    return $people;
                }

                return false;
            }));

            if(false === $personType || (preg_match('/(courtesy)/i', $personType['displayName']) && !$this->accessChecker->courtesyIsEnabled())) {
                return [
                  'page' => [
                    '#theme' => 'orglist-faculty',
                    '#faculty' => null,
                    '#imageData' => null,
                  ]
                ];
            }

            $facultyData = array_filter($facultyData, static function($value) use($personType) {

               if($value['type'] === $personType['displayName']) {
                   return $value;
               }
               return false;
            });
        }


        if(($areaId = $this->request->query->get('areas')) && mb_strtolower($areaId) !== 'all') {
            $facultyData = array_filter($facultyData, function($value) use($areaId) {
                foreach($value['areas'] as $area) {

                    $areaConfigItem = sprintf('include_%s_areas', $area['type']);
                    $areaIsEnabled = (bool)($this->config('orglist.settings')->get($areaConfigItem) ?? TRUE);
                    if($area['id'] === $areaId && (bool)$area['enabled'] && (bool)$areaIsEnabled) {
                        return $value;
                    }
                }
                return false;
            });
        }

        $facultyCount = \count($facultyData);


        $args = $this->buildFacultyFormArgs();

        $args['count'] = $facultyCount;

        $sortForm = \Drupal::formBuilder()->getForm(OrglistFacultySortForm::class, $args);

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();
        $facultyData = Utils::sortArrayBy($facultyData, 'lastName', $direction);

        if(false === $this->request->query->has('type') || $this->request->query->get('type') === 'active') {
            $facultyData = array_filter($facultyData, static function ($value) {
                if (preg_match('/(courtesy|visiting|visitor|emeritus)/i',
                  $value['type'])) {
                    return false;
                }
                return $value;
            });
        }

        $data = array_chunk($facultyData, $limit);
        $pager = $this->pagerManager->createPager($facultyCount, $limit);

        return [
          'sort_form' => [
            'sort_form' => $sortForm,
          ],
            'page' => [
              '#theme' => 'orglist-faculty',
              '#faculty' => $data[$pager->getCurrentPage()],
              '#imageData' => $imageData,
            ],
            'pager' => [
              '#type' => 'pager',
                '#quantity' => 5,
            ],
        ];
    }

    #[ArrayShape([
      'sort_form' => "array",
      'page'      => "array",
      'pager'     => "array"
    ])]
    public function visitors(): array
    {
        if(!$this->accessChecker->visitorIsEnabled()) {
            throw new NotFoundHttpException();
        }

        $visitors = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_VISITOR);

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();

        $visitors = Utils::sortArrayBy($visitors, 'lastName', $direction);
        $visitorCount = \count($visitors);

        $form = \Drupal::formBuilder()->getForm(OrglistBaseSortForm::class, ['count' => $visitorCount]);

        $pages = array_chunk($visitors, $limit);
        $pager = $this->pagerManager->createPager($visitorCount, $limit);

        return [
            'sort_form' => [
              'sort_form' => $form,
            ],
            'page' => [
                '#theme' => 'orglist-visitor',
                '#visitors' => \count($pages) > 0 ?  $pages[$pager->getCurrentPage()] : null,
            ],
            'pager' => [
                '#type' => 'pager',
                '#quantity' => 5,
            ]
        ];
    }

    #[ArrayShape([
      'sort_form' => "array",
      'page'      => "array",
      'pager'     => "array"
    ])]
    public function student(): array
    {
        if(!$this->accessChecker->studentIsEnabled()) {
            throw new NotFoundHttpException();
        }

        $students = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_STUDENT);
        $studentCount = \count($students);

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();

        $students = Utils::sortArrayBy($students, 'lastName', $direction);

        $pages = array_chunk($students, $limit);
        $pager = $this->pagerManager->createPager(\count($students), $limit);

        $form = \Drupal::formBuilder()->getForm(OrglistBaseSortForm::class, ['count' => $studentCount]);

        return [
          'sort_form' => [
            'sort_form' => $form,
          ],
          'page' => [
            '#theme' => 'orglist-student',
            '#student' => $pages[$pager->getCurrentPage()],
          ],
          'pager' => [
            '#type' => 'pager',
            '#quantity' => 5,
          ]
        ];
    }

    #[ArrayShape([
      'sort_form' => "array",
      'page'      => "array",
      'pager'     => "array"
    ])]
    public function postDoc(): array
    {
        if(!$this->accessChecker->postDocIsEnabled()) {
            throw new NotFoundHttpException();
        }

        $postDoc = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_POSTDOC);
        $postDocCount = \count($postDoc);

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();

        $postDoc = Utils::sortArrayBy($postDoc, 'lastName', $direction);

        $form = \Drupal::formBuilder()->getForm(OrglistBaseSortForm::class, ['count' => $postDocCount]);

        $pages = array_chunk($postDoc, $limit);
        $pager = $this->pagerManager->createPager($postDocCount, $limit);

        return [
            'sort_form' => [
                'sort_form' => $form,
            ],
          'page' => [
            '#theme' => 'orglist-postdoc',
            '#postdoc' => $pages[$pager->getCurrentPage()],
          ],
          'pager' => [
            '#type' => 'pager',
            '#quantity' => 5,
          ]
        ];
    }

    #[ArrayShape([
      'sort_form' => "array",
      'page'      => "array",
      'pager'     => "array"
    ])]
    public function adjunct(): array
    {
        if(!$this->accessChecker->consultingIsEnabled()) {
            throw new NotFoundHttpException();
        }

        $lecturersAreEnabled = $this->accessChecker->lecturerIsEnabled();

        $consultingFaculty = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_CONSULTING);

        $lecturers = [];

        if($lecturersAreEnabled) {
            $lecturers = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_LECTURER);
        }

        $consultingFaculty = array_map(static function($value) {
            $value['type'] = str_ireplace('consulting', 'Adjunct', $value['type']);
            return $value;
        }, $consultingFaculty);

        $adjunct = array_merge($consultingFaculty, $lecturers);

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();

        $adjunct = Utils::sortArrayBy($adjunct, 'lastName', $direction);

        $sortForm = \Drupal::formBuilder()->getForm(OrglistBaseSortForm::class, ['count' => \count($adjunct)]);

        $results = array_chunk($adjunct, $limit);

        $pager = $this->pagerManager->createPager(\count($adjunct), $limit);

        $lecturersImageData = [];

        $consultingImageData = $this->cacheUtility->getImageCache(CacheUtility::CACHE_KEY_IMAGE_CONSULTING);

        if($lecturersAreEnabled) {
            $lecturersImageData = $this->cacheUtility->getImageCache(CacheUtility::CACHE_KEY_IMAGE_LECTURER);
        }

        $imageData = array_merge($consultingImageData, $lecturersImageData);

        return [
          'sort_form' => [
            'sort_form' => $sortForm,
          ],
          'page' => [
            '#theme' => 'orglist-faculty',
            '#faculty' => $results[$pager->getCurrentPage()],
              '#imageData' => $imageData,
          ],
            'pager' => [
              '#type' => 'pager',
              '#quantity' => 5
            ]
        ];
    }

    /**
     * @return array
     *   Defaults are ['direction' => 'asc', 'limit' => 10]
     */
    #[ArrayShape([
      'direction' => "string",
      'limit' => "int"
    ])]
    private function getDirectionAndLimit(): array
    {
        return [
            'direction' => $this->request->query->get('sort', 'asc'),
            'limit' => $this->request->query->getInt('limit', 10),
        ];
    }

    #[ArrayShape([
      'sort_form' => "array",
      'page'      => "array",
      'pager'     => "array"
    ])]
    public function staff(): array
    {
        if(!$this->accessChecker->staffIsEnabled()) {
            throw new NotFoundHttpException();
        }

        $staff = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_STAFF);

        $personTypes = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_PERSON_TYPE);

        $staffType = current(array_filter($personTypes, static function($value) {
           if(mb_strtolower($value['displayName']) === 'admin/research staff') {
               return $value;
           }
           return false;
        }));

        $staffTypes = [];
        $staffTypes['all'] = 'All';
        $staffTypes[$staffType['id']] = $staffType['displayName'];
        foreach($staffType['subTypes'] as $subType) {
            $staffTypes[$subType['id']] = $subType['displayName'];
        }

        ['direction' => $direction, 'limit' => $limit] = $this->getDirectionAndLimit();

        $staff = $this->filterType($staffTypes, $staff);

        $staff = Utils::sortArrayBy($staff, 'lastName', $direction);

        $staffCount = \count($staff);
        $form = \Drupal::formBuilder()->getForm(OrglistStaffSortForm::class, ['count' => $staffCount, 'staff_types' => $staffTypes]);

        $pager = $this->pagerManager->createPager($staffCount, $limit);
        $results = array_chunk($staff, $limit);


        return [
          'sort_form' => [
            'sort_form' => $form,
          ],
          'page' => [
            '#theme' => 'orglist-staff',
            '#staff' => $results[$pager->getCurrentPage()],
            '#imageData' => $this->cacheUtility->getImageCache(CacheUtility::CACHE_KEY_IMAGE_STAFF),
          ],
            'pager' => [
              '#type' => 'pager',
              '#quantity' => 5,
            ]
        ];
    }

    /**
     * @param array $types
     * @param array $people
     *
     * @return array
     */
    protected function filterType(array $types, array $people): array
    {
        if ($this->request->query->has('type') && $this->request->query->get('type') !== 'all') {
            $type = $this->request->query->get('type');
            $typeName = $types[$type];
            return array_filter($people,
              static function ($person) use ($typeName) {
                  if ($typeName === $person['type']) {
                      return $person;
                  }
                  return false;
              });
        }

        return $people;
    }

    public function getPeopleTitle(): string
    {
        return $this->config('orglist.settings')->get('people_page_title') ?? 'People';
    }

    /**
   * Builds the response.
   */
  #[ArrayShape(['#theme'   => "string",
                '#content' => "string|null",
                '#title'   => "string|null"
  ])]
  public function people(): array {

      return [
          '#theme' => 'orglist-people',
          '#content' => $this->config('orglist.settings')->get('people_page_content'),
          '#title' => $this->config('orglist.settings')->get('people_page_title'),
      ];

  }

 /*
  * array:18 [▼
  'id' => '0be6775a-a0fa-4a98-88cc-4538073360c1'
  'type' => 'Associate Professor'
  'adminTitle' => null
  'department' => null
  'displayName' => 'Amin Arbabian'
  'email' => 'arbabian@stanford.edu'
  'firstName' => 'Amin'
  'lastName' => 'Arbabian'
  'location' => 'Allen 204'
  'phone' => null
  'psName' => 'Amin Arbabian'
  'sunetId' => 'arbabian'
  'website' => 'https://arbabianlab.stanford.edu/'
  'mailCode' => '4070'
  'areas' => array:9 [▼
    0 => array:4 [▼
      'id' => '068ce5f6-5b8b-4550-ad77-e64c264c9946'
      'type' => 'research'
      'name' => 'Hardware/Software Systems'
      'secondaryName' => null
    ]
    1 => array:4 [▼
      'id' => '1de6f81c-508a-4465-ac50-5379a5811f21'
      'type' => 'subarea'
      'name' => 'Integrated Circuits & Power Electronics'
      'secondaryName' => null
    ]
    2 => array:4 [▶]
    3 => array:4 [▶]
    4 => array:4 [▶]
    5 => array:4 [▶]
    6 => array:4 [▶]
    7 => array:4 [▶]
    8 => array:4 [▶]
  ]
  'groups' => array:9 [▼
    0 => array:4 [▼
      'id' => '068ce5f6-5b8b-4550-ad77-e64c264c9946'
      'type' => 'research'
      'name' => 'Hardware/Software Systems'
      'secondaryName' => null
    ]
    1 => array:4 [▶]
    2 => array:4 [▶]
    3 => array:4 [▶]
    4 => array:4 [▶]
    5 => array:4 [▶]
    6 => array:4 [▶]
    7 => array:4 [▶]
    8 => array:4 [▶]
  ]
  'facultyAdmin' => array:8 [▼
    'id' => 'ef77fcb4-40e9-4d2d-8ab3-efe187e50d6b'
    'name' => 'Niu, Helen'
    'email' => 'helen.niu@stanford.edu'
    'location' => 'Packard 310'
    'mailCode' => '9510'
    'phone' => '723-8121'
    'areas' => []
    'groups' => []
  ]
  'researchAdmin' => null
]
  *//**
 * @param array $args
 * @return array
 */
    protected function buildFacultyFormArgs(): array
    {
        $args = [];
        $args['rowClass'] = 'row-cols-md-3';

        $args = $this->buildPersonTypeFacultyFormArgs($args);
        return $this->buildAreaFacultyFormArgs($args);
    }

    protected function buildPersonTypeFacultyFormArgs(array $args): array
    {
        $personTypes = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_PERSON_TYPE);

        $professorTypes = array_values(array_filter($personTypes,
          function ($value) {
              if (mb_strtolower($value['displayName']) === 'professor' || (mb_strtolower($value['displayName']) === 'courtesy professor' && $this->accessChecker->courtesyIsEnabled())) {
                  return $value;
              }
              return false;
          }));

        if (\count($professorTypes) > 0) {

            $filteredProfessorTypes = [];
            $filteredProfessorTypes['all'] = $this->t('All');
            $filteredProfessorTypes['active'] = $this->t('Active Faculty');
            foreach ($professorTypes as $professorType) {
                $filteredProfessorTypes[$professorType['id']] = $professorType['displayName'];
                if (isset($professorType['subTypes'])) {
                    foreach ($professorType['subTypes'] as $subType) {
                        $filteredProfessorTypes[$subType['id']] = $subType['displayName'];
                    }
                }
            }

            if($this->accessChecker->courtesyIsEnabled()) {
                uasort($filteredProfessorTypes,
                  static function ($value, $value2) {
                      if (preg_match('/(courtesy)/i', $value)) {
                          return 1;
                      }
                      return -1;
                  });
            }

            $args['professorTypes'] = $filteredProfessorTypes;
            $args['rowClass'] = 'row-cols-md-4';
        }

        return $args;
    }

    protected function buildAreaFacultyFormArgs(array $args): array
    {
        $orgGroups = $this->cacheUtility->getCache(CacheUtility::CACHE_KEY_ORG_GROUP);

        $orgGroupOptions = [
          'all' => $this->t('All'),
        ];


        $settings = $this->config('orglist.settings');

        $includeBuildingAreas = (bool)($settings->get('include_building_areas') ?? FALSE);
        $includeSubAreas = (bool)($settings->get('include_sub_areas') ?? TRUE);
        $includeResearchAreas = (bool)($settings->get('include_research_areas') ?? TRUE);

        foreach($orgGroups as $orgGroup) {

            $type =  match($orgGroup['type']) {
                default => 'none',
                'area' => 'Building Area',
                'subarea' => 'Sub Areas',
                'research' => 'Research Areas',
            };

            if( !(bool)($orgGroup['enabled'] ?? true) ) {
              continue;
            }

            if($orgGroup['type'] === 'area' && !$includeBuildingAreas) {
                continue;
            }

            if($orgGroup['type'] === 'subarea' && !$includeSubAreas) {
              continue;
            }

            if($orgGroup['type'] === 'research' && !$includeResearchAreas) {
              continue;
            }

          $orgGroupOptions[$type][$orgGroup['id']] = $orgGroup['name'];
        }

        if(count($orgGroupOptions) === 2) {
          $oneType = $orgGroupOptions[array_key_last($orgGroupOptions)];
          array_pop($orgGroupOptions);
          uasort($oneType, static fn($value1, $value2) => $value1 <=> $value2);
          $orgGroupOptions += $oneType;

        }
        foreach($orgGroupOptions as &$orgGroupArray) {
          if(!is_array($orgGroupArray)) { continue; }
          uasort($orgGroupArray, static fn($value1, $value2) => $value1 <=> $value2);
        }
        unset($orgGroupArray);

        $args['orgGroupOptions'] = $orgGroupOptions;

        return $args;
    }
}
