<?php

namespace Drupal\orglist\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Render\Element\MachineName;
use Drupal\node\Entity\Node;
use Drupal\orglist\Cache\CacheUtility;
use Drupal\orglist\Request\Request;
use Drupal\orglist\Utils;
use Drupal\system\MachineNameController;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Orglist settings for this site.
 */
class SettingsForm extends ConfigFormBase {

    public const SETTINGS_PERSON_TYPE_FORMAT = 'enable_%s_type';

    private array $personTypes;
    private array $personTypeKeys = [];
    private MenuLinkManagerInterface $menuLinkManager;
    private CacheBackendInterface $cacheMenu;

    public function __construct(ConfigFactoryInterface $config_factory, MenuLinkManagerInterface $menuLinkManager, CacheBackendInterface $cacheMenu)
    {
        parent::__construct($config_factory);
        $this->personTypes = (new CacheUtility())->getCache(CacheUtility::CACHE_KEY_PERSON_TYPE);
        $this->menuLinkManager = $menuLinkManager;
        $this->cacheMenu = $cacheMenu;
    }

    public static function create(ContainerInterface $container)
    {
        return new static(
         $container->get('config.factory'),
          $container->get('plugin.manager.menu.link'),
          $container->get('cache.menu')
        );
    }

    /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'orglist_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['orglist.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {

    $orglistSettings = $this->config('orglist.settings');

    $form['api_url'] = [
        '#type' => 'url',
        '#title' => $this->t('Endpoint for the Orglist API.'),
        '#default_value' => $orglistSettings->get('api_url'),
    ];

    $form['cache_lifetime'] = [
        '#type' => 'number',
        '#title' => $this->t('The Orglist cache lifetime. (in minutes)'),
        '#description' => $this->t('How often the cache should be refreshed automatically.'),
        '#default_value' => (int)($orglistSettings->get('cache_lifetime') ?? 60),
        '#size' => 5,
        '#min' => 15,
        '#step' => 5,
    ];

    $form['flush_cache_with_drupal'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Flush cache with drupal cache.'),
        '#description' => $this->t('When flushing all drupal cache, should we flush the orglist cache?'),
        '#default_value' => $orglistSettings->get('flush_cache_with_drupal') ?? false,
    ];

    $form['flush_image_cache_with_drupal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Flush image cache with drupal cache.'),
      '#description' => $this->t('When flushing all drupal cache, should we flush the orglist image cache?'),
      '#default_value' => $orglistSettings->get('flush_image_cache_with_drupal') ?? false,
    ];

    $form['show_short_faculty_list'] = [
      '#type' => 'checkbox',
        '#title' => $this->t('Show a short list of faculty on research pages? (Not Implemented)'),
        '#description' =>$this->t('This will enable a short list of faculty associated with the research page.'),
        '#default_value' => $orglistSettings->get('show_short_faculty_list') ?? true,
    ];

    $form['include_research_areas'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable research areas in faculty lists.'),
      '#description' => $this->t('This will enable or disable research areas in faculty lists.'),
      '#default_value' => $orglistSettings->get('include_research_areas') ?? true,
    ];

    $form['include_subarea_areas'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable sub areas in faculty lists.'),
      '#description' => $this->t('This will enable or disable sub areas in faculty lists.'),
      '#default_value' => $orglistSettings->get('include_subarea_areas') ?? true,
    ];

    $form['include_area_areas'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable building areas in faculty lists.'),
      '#description' => $this->t('This will enable or disable building areas in faculty lists. (Packard, Allen, etc.)'),
      '#default_value' => $orglistSettings->get('include_area_areas') ?? false,
    ];

    if(null !== $orglistSettings->get('disabled_pages')) {

      $disabledNodeIds = [];
      foreach ($orglistSettings->get('disabled_pages') as $value) {
        $disabledNodeIds[] = (int) $value['target_id'];
      }

      $disabledPages = Node::loadMultiple($disabledNodeIds);
    }

    $form['disabled_pages'] = [
        '#type' => 'entity_autocomplete',
        '#title' => $this->t('Disabled research pages.'),
        '#description' => $this->t('Pages that should be excluded from displaying short faculty info.'),
        '#target_type' => 'node',
        '#default_value' => $disabledPages ?? null, // The #default_value can be either an entity object or an array of entity objects.
        '#tags' => true,
    ];

    $form['people_page_title'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Title for the /people page.'),
        '#default_value' => $orglistSettings->get('people_page_title'),
    ];

    $peoplePageContentConfig = $orglistSettings->get('people_page_content');

   $form['people_page_content'] = [
     '#title' => $this->t('Content for the people landing page.'),
     '#description' => $this->t('The content that should appear when users navigate to /people'),
     '#type' => 'text_format',
     '#format' => $peoplePageContentConfig['format'] ?? filter_default_format(),
     '#rows' => 10,
     '#default_value' => $peoplePageContentConfig['value'] ?? '',
   ];

      $options = [];

      $getWeight = static function ($type)
      {
          if($type === 'Professor') {
              return -1;
          }

          if($type === 'Admin/Research Staff') {
              return 0;
          }
          return mb_strlen($type);
      };
      if(!empty($this->personTypes)) {
        $form['person_types'] = [
          '#type'        => 'details',
          '#title'       => $this->t('Person types that should be enabled from orglist.'),
          '#description' => $this->t('Person types that should be enabled on the website.'),
          '#open'        => TRUE,
        ];

        foreach ($this->personTypes as $personType) {
          if (empty($personType['subTypes'])) {
            continue;
          }
          $personTypeKey = sprintf(self::SETTINGS_PERSON_TYPE_FORMAT,
            Utils::getMachineName($personType['displayName']));

          $this->personTypeKeys[$personType['id']] = $personTypeKey;

          $form['person_types'][$personTypeKey] = [
            '#type'          => 'checkbox',
            '#title'         => $this->t('Enable @personType',
              ['@personType' => $personType['displayName']]),
            '#description'   => $this->t('Enable @personType type from orglist on website.',
              ['@personType' => $personType['displayName']]),
            '#default_value' => $orglistSettings->get($personTypeKey) ?? TRUE,
            '#weight'        => $getWeight($personType['displayName']),
          ];
        }

        $form['person_types']['enable_student_type'] = [
          '#type'          => 'checkbox',
          '#title'         => $this->t('Enable Students'),
          '#description'   => $this->t('Enable Student from orglist on website.'),
          '#default_value' => $orglistSettings->get('enable_student_type') ?? TRUE,
          '#weight'        => $getWeight('Student'),
        ];

        $this->personTypeKeys['student'] = 'enable_student_type';
      }

      $form['authentication'] = [
        '#type' => 'details',
        '#open' => true,
        '#title' => $this->t('Authentication options'),
        '#description' => $this->t('Authentication options for the Orglist API and CAP API (if used).')
      ];
      $form['authentication']['api_token'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Orglist API token.'),
        '#default_value' => $this->config('orglist.settings')->get('api_token'),
      ];

      $form['authentication']['api_device'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Orglist API device name.'),
        '#default_value' => $this->config('orglist.settings')->get('api_device'),
      ];

      $form['authentication']['cap'] = [
        '#type' => 'details',
        '#open' => false,
        '#title' => $this->t('Stanford CAP API options.'),
      ];

      $form['authentication']['cap']['use_cap'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Use CAP API to retrieve staff and faculty images?'),
        '#description' => $this->t('If this is checked the module will use the CAP API to retrieve images.  Otherwise it will scrape profile.stanford.edu pages for the images.'),
        '#default_value' => $this->config('orglist.settings')->get('use_cap') ?? true,
      ];

      $form['authentication']['cap']['cap_api_username'] = [
        '#type' => 'textfield',
        '#title' => $this->t('CAP API user name.'),
        '#default_value' => $this->config('orglist.settings')->get('cap_api_username'),
      ];

      $form['authentication']['cap']['cap_api_password'] = [
        '#type' => 'password',
        '#title' => $this->t('CAP API password.'),
        '#default_value' => $this->config('orglist.settings')->get('cap_api_password'),
      ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

      $apiUrl = $form_state->getValue('api_url');
      if( false === UrlHelper::isValid($apiUrl) ) {
         $form_state->setErrorByName('api_url', $this->t('The Url @url is not valid.', ['@url' => $apiUrl]));
     }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    $config = $this->configFactory->getEditable('orglist.settings');
    $config
      ->set('api_url', $form_state->getValue('api_url'))
      ->set('api_token', $form_state->getValue('api_token'))
      ->set('api_device', $form_state->getValue('api_device'))
      ->set('people_page_title', $form_state->getValue('people_page_title'))
      ->set('people_page_content', $form_state->getValue('people_page_content'))
      ->set('show_short_faculty_list', $form_state->getValue('show_short_faculty_list'))

      ->set('include_research_areas', $form_state->getValue('include_research_areas'))
      ->set('include_subarea_areas', $form_state->getValue('include_subarea_areas'))
      ->set('include_area_areas', $form_state->getValue('include_area_areas'))

      ->set('cache_lifetime', (int)$form_state->getValue('cache_lifetime'))
      ->set('disabled_pages', $form_state->getValue('disabled_pages'))
      ->set('flush_cache_with_drupal', $form_state->getValue('flush_cache_with_drupal'))
      ->set('flush_image_cache_with_drupal', $form_state->getValue('flush_image_cache_with_drupal'))
      // cap
      ->set('use_cap', $form_state->getValue('use_cap'))
      ->set('cap_api_username', $form_state->getValue('cap_api_username'))
      ->save();

      if(!$form_state->isValueEmpty('cap_api_password')) {
          $config
            ->set('cap_api_password', $form_state->getValue('cap_api_password'))
            ->save();
      }

    foreach($this->personTypeKeys as $personTypeKey) {
        $config
            ->set($personTypeKey, $form_state->getValue($personTypeKey))
            ->save();

    }
      $this->cacheMenu->invalidateAll();
      $this->menuLinkManager->rebuild();

    parent::submitForm($form, $form_state);
  }

}
