<?php

namespace Drupal\orglist\Request;

use Drupal;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\orglist\Http\OrglistClientFactory;
use Drupal\orglist\Utils as OrglistUtils;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Promise\Utils as GuzzlePromiseUtils;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use ReflectionClassConstant;

class Request
{

    /*
     *  List of Orglist endpoints
     * ----------------------------------------------------------------------------------------
     *  Name                       Method   Scheme   Host   Path
     * ------------------------ -------- -------- ------ --------------------------------------
     *  consulting_faculty_list    GET      ANY      ANY    /orglist/v1/consultingFaculty
     *  one_consulting_faculty     GET      ANY      ANY    /orglist/v1/consultingFaculty/{id}
     *  courtesy_faculty_list      GET      ANY      ANY    /orglist/v1/courtesyFaculty
     *  one_courtesy_faculty       GET      ANY      ANY    /orglist/v1/courtesyFaculty/{id}
     *  admin_list                 GET      ANY      ANY    /orglist/v1/admin
     *  one_admin                  GET      ANY      ANY    /orglist/v1/admin/{id}
     *  faculty_list               GET      ANY      ANY    /orglist/v1/faculty
     *  one_faculty                GET      ANY      ANY    /orglist/v1/faculty/{id}
     *  lecturer_list              GET      ANY      ANY    /orglist/v1/lecturer
     *  one_lecturer               GET      ANY      ANY    /orglist/v1/lecturer/{id}
     *  org_group_list             GET      ANY      ANY    /orglist/v1/orgGroup
     *  org_group_by_type          GET      ANY      ANY    /orglist/v1/orgGroup/{areaType}
     *  org_group_by_type_and_id   GET      ANY      ANY    /orglist/v1/orgGroup/{areaType}/{id}
     *  person_type_list           GET      ANY      ANY    /orglist/v1/personType
     *  one_person_type            GET      ANY      ANY    /orglist/v1/personType/{id}
     *  postDoc_list               GET      ANY      ANY    /orglist/v1/postDoc
     *  one_postDoc                GET      ANY      ANY    /orglist/v1/postDoc/{id}
     *  staff_list                 GET      ANY      ANY    /orglist/v1/staff
     *  one_staff                  GET      ANY      ANY    /orglist/v1/staff/{id}
     *  student_list               GET      ANY      ANY    /orglist/v1/student
     *  one_student                GET      ANY      ANY    /orglist/v1/student/{id}
     *  visitor_list               GET      ANY      ANY    /orglist/v1/visitor
     *  one_visitor                GET      ANY      ANY    /orglist/v1/visitor/{id}
     */

    public const ENDPOINT_CONSULTING = 'consultingFaculty';
    public const ENDPOINT_COURTESY = 'courtesyFaculty';
    public const ENDPOINT_ADMIN = 'admin';
    public const ENDPOINT_FACULTY = 'faculty';
    public const ENDPOINT_LECTURER = 'lecturer';
    public const ENDPOINT_ORG_GROUP = 'orgGroup';
    public const ENDPOINT_POSTDOC = 'postDoc';
    public const ENDPOINT_STAFF = 'staff';
    public const ENDPOINT_STUDENT = 'student';
    public const ENDPOINT_VISITOR = 'visitor';
    public const ENDPOINT_PERSON_TYPE = 'personType';

    use StringTranslationTrait;
    use LoggerChannelTrait;

    private Client $client;
    private LoggerInterface $logger;
    private ImmutableConfig $orglistConfig;

    public function __construct()
    {
        $this->orglistConfig = Drupal::config('orglist.settings');

        $this->client = OrglistClientFactory::get($this->orglistConfig->get('api_device'),
          $this->orglistConfig->get('api_token'));
        $this->logger = $this->getLogger('orglist');
    }

    public function requestAll(): array
    {
        if (null === $this->orglistConfig->get('api_token')) {
            \Drupal::messenger()
              ->addError($this->t('Please set the orglist api token in the configuration section.'));
            $this->logger->error($this->t('The orglist api token has not been set yet.'));
            return [];
        }

        $requests = [];

        $selfReflection = new ReflectionClass($this); // We could all use a little more of this.
        $this->logger->debug($this->t('Checking enabled types ...'));
        foreach ($selfReflection->getConstants(ReflectionClassConstant::IS_PUBLIC) as $endpoint) {
            $this->logger->debug($this->t('Checking if the endpoint "@endpoint" should be enabled.',
              ['@endpoint' => $endpoint]));
            if (OrglistUtils::checkEnabledByCid($endpoint)) {
                $this->logger->debug($this->t('Adding the endpoint "@endpoint" to the request queue.',
                  ['@endpoint' => $endpoint]));
                $requests[$endpoint] = $this->client->requestAsync('GET',
                  $endpoint);
            }
        }

        return GuzzlePromiseUtils::settle($requests)->wait();

    }

    public function request(string $endpoint): ResponseInterface
    {
        if (null === $this->orglistConfig->get('api_token')) {
            \Drupal::messenger()
              ->addError($this->t('Please set the orglist api token in the configuration section.'));
            $this->logger->error($this->t('The orglist api token has not been set yet.'));
            return new Response(424, [], null, '1.1',
              $this->t('Failed Dependency, The orglist api token has not been set yet.'));
        }

        try {
            $this->logger->debug($this->t('Requesting data for endpoint "@endpoint".',
              ['@endpoint' => $endpoint]));
            return $this->client->request('GET', $endpoint);
        } catch (GuzzleException $e) {
            $this->logger->error(
              $this->t(
                'Unable to get data for endpoint "@endpoint".  Message was : @message.',
                [
                  '@endpoint' => $endpoint,
                  '@message'  => $e->getMessage(),
                ])
            );
            return new Response($e->getCode() === 0 ? 500 : $e->getCode(), [], null, '1.1', $e->getMessage());
        }
    }
}