<?php


namespace Drupal\orglist\Cache;

use Drupal;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\orglist\Request\Request;
use Drupal\orglist\Utils as OrglistUtils;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Response;
use JetBrains\PhpStorm\ArrayShape;
use Psr\Log\LoggerInterface;
use Suee\CAP\Authorization\Token;
use Suee\CAP\Authorization\TokenError;
use Suee\CAP\Authorization\TokenFactory;
use Suee\CAP\Request\ImageRequest;
use function array_key_exists;
use function implode;
use function mb_strtolower;
use function sprintf;
use function str_contains;
use function str_replace;

class CacheUtility {

  public const CACHE_KEY_PEOPLE = 'people';
  public const CACHE_KEY_CONSULTING = self::CACHE_KEY_PEOPLE . ':consultingFaculty';
  public const CACHE_KEY_COURTESY = self::CACHE_KEY_PEOPLE . ':courtesyFaculty';
  public const CACHE_KEY_ADMIN = self::CACHE_KEY_PEOPLE . ':admin';
  public const CACHE_KEY_FACULTY = self::CACHE_KEY_PEOPLE . ':faculty';
  public const CACHE_KEY_LECTURER = self::CACHE_KEY_PEOPLE . ':lecturer';
  public const CACHE_KEY_ORG_GROUP = self::CACHE_KEY_PEOPLE . ':orgGroup';
  public const CACHE_KEY_POSTDOC = self::CACHE_KEY_PEOPLE . ':postDoc';
  public const CACHE_KEY_STAFF = self::CACHE_KEY_PEOPLE . ':staff';
  public const CACHE_KEY_STUDENT = self::CACHE_KEY_PEOPLE . ':student';
  public const CACHE_KEY_VISITOR = self::CACHE_KEY_PEOPLE . ':visitor';
  public const CACHE_KEY_PERSON_TYPE = self::CACHE_KEY_PEOPLE . ':personType';

  public const CACHE_KEY_IMAGE_BASE = 'image';
  public const CACHE_KEY_IMAGE_FACULTY = self::CACHE_KEY_IMAGE_BASE . ':faculty';
  public const CACHE_KEY_IMAGE_COURTESY = self::CACHE_KEY_IMAGE_BASE . ':courtesyFaculty';
  public const CACHE_KEY_IMAGE_CONSULTING = self::CACHE_KEY_IMAGE_BASE . ':consultingFaculty';
  public const CACHE_KEY_IMAGE_STAFF = self::CACHE_KEY_IMAGE_BASE . ':staff';
  public const CACHE_KEY_IMAGE_LECTURER = self::CACHE_KEY_IMAGE_BASE . ':lecturer';

  public const DEFAULT_IMAGE_URL = 'https://profiles.stanford.edu/images/placeholder-350x522.png';

  private CacheBackendInterface $cache;
  private CacheBackendInterface $imageCache;
  private Request $request;
  private LoggerInterface $logger;

  use StringTranslationTrait;

  public function __construct()
  {
    $this->cache = Drupal::cache('orglist');
    $this->imageCache = Drupal::cache('orglist_image');
    $this->request = new Request();
    $this->logger = Drupal::logger('orglist');
  }

  #[ArrayShape([
      'faculty'    => "array",
      'staff'      => "array",
      'consulting' => "array",
      'lecturer'   => "array",
      'courtesy'   => "array",
    ]
  )]
  public function rebuildAllImageCache(): array
  {
    $this->imageCache->deleteAll();
    $faculty = $this->getImageCache(self::CACHE_KEY_IMAGE_FACULTY);
    $staff = $this->getImageCache(self::CACHE_KEY_IMAGE_STAFF);
    $consulting = $this->getImageCache(self::CACHE_KEY_IMAGE_CONSULTING);
    $courtesy = $this->getImageCache(self::CACHE_KEY_IMAGE_COURTESY);
    $lecturer = $this->getImageCache(self::CACHE_KEY_IMAGE_LECTURER);

    return [
      'faculty'    => $faculty,
      'staff'      => $staff,
      'consulting' => $consulting,
      'lecturer'   => $lecturer,
      'courtesy'   => $courtesy,
    ];

  }

  public function getImageCache($cid): array
  {
    if (!Drupal::config('orglist.settings')->get('use_cap')) {
      Drupal::messenger()->addError(
        $this->t('To use images from cap you need to enable cap in the settings and enter your cap api username and password.')
      );
      $this->logger->error($this->t('To use cap images, enable them in the orglist configuration settings.'));
      return [];
    }

    $cacheKey = match ($cid) {
      self::CACHE_KEY_IMAGE_FACULTY => self::CACHE_KEY_FACULTY,
      self::CACHE_KEY_IMAGE_STAFF => self::CACHE_KEY_STAFF,
      self::CACHE_KEY_IMAGE_COURTESY => self::CACHE_KEY_COURTESY,
      self::CACHE_KEY_IMAGE_CONSULTING => self::CACHE_KEY_CONSULTING,
      self::CACHE_KEY_IMAGE_LECTURER => self::CACHE_KEY_LECTURER,
      default => [],
    };

    if ($cacheKey === []) {
      $this->logger->error($this->t('Invalid image cache key "@cacheKey".',
        ['@cacheKey' => $cid]));
      return $cacheKey;
    }

    if ($imageCache = $this->imageCache->get($cid)) {
      $this->logger->info($this->t('Found image cache for @cid.',
        ['@cid' => $cid]));
      return $imageCache->data;
    }

    $this->logger->info($this->t('Missed image cache for @cid, generating new.',
      ['@cid' => $cid]));

    if (NULL === $token = $this->getCapToken()) {
      return $this->imageCache->get($cid, TRUE)->data;
    }

    $capApi = new ImageRequest($token);
    // Set per page results, maybe make the batch size configurable in settings.
    $capApi->setQueryParam('ps', 20);
    $uids = $this->generateUids($cacheKey);

    $images = [];
    // Maybe make batch size configurable in settings.
    foreach (array_chunk($uids, 20) as $batch) {
      $uidQueryString = implode(',', $batch);
      $capApi->setQueryParam('uids', $uidQueryString);

      if (NULL !== $result = $capApi->getRequest()
          ?->getBody()
          ->getContents()) {

        try {
          $array = \GuzzleHttp\json_decode($result, TRUE);
          if (isset($array['error'], $array['error_description'])) {
            $this->logger->error(
              $this->t(
                'Error retrieving images. Error was @error, @errorDescription',
                [
                  '@error'            => $array['error'],
                  '@errorDescription' => $array['error_description'],
                ]
              )
            );
            return $this->imageCache->get($cid, TRUE)->data;
          }
        } catch (Exception $e) {
          $this->logger->error(
            $this->t('Unable to get image cache for "@uids". (@message).',
              [
                '@uids'    => $uidQueryString,
                '@message' => $e->getMessage(),
              ])
          );
          continue;
        }
        /*
         * CAP API structure
         * ------------------
         * [root]['values'][int]['profilePhotos']['bigger']['url'];
         * [root]['values'][int]['uid'];
         * [root]['values'][int]['displayName']
         * [root]['values'][int]['names']['preferred'] (['firstName'], ['lastName'], ['middleName'])
         */
        foreach ($array['values'] as $person) {
          $preferredName = str_replace('  ', ' ', sprintf(
            '%s %s %s',
            $person['names']['preferred']['firstName'] ?? '',
            $person['names']['preferred']['middleName'] ?? '',
            $person['names']['preferred']['lastName'] ?? ''
          ));

          $imageUrl = sprintf('https://profiles.stanford.edu/proxy/api/cap/profiles/%d/resources/profilephoto/350x522.%d.jpg',
            (int) $person['profileId'],
            Drupal::time()->getRequestTime());

          $this->logger->info(
            $this->t(
              'Setting image for @preferredName to @imageUrl',
              [
                '@preferredName' => $preferredName,
                '@imageUrl'      => $imageUrl,
              ]
            )
          );

          $images[$person['uid']] = $imageUrl;

        }
      }
    }

    $this->setImageCache($cid, $images);

    return $images;

  }

  public function setImageCache(string $cid, array $images): void {
    $this->imageCache->set($cid, $images);
  }

  private function getCapToken(): ?Token {

    $config = Drupal::config('orglist.settings');

    $tokenFactory = new TokenFactory(
      $config->get('cap_api_username'),
      $config->get('cap_api_password')
    );

    $token = $tokenFactory->request();

    if ($token?->hasErrors()) {
      $error = $token?->getErrors();
      $message = $this->t('Unknown error occurred retrieving cap token.');

      if ($error instanceof TokenError) {
        $message = $this->t(
          'Error retrieving cap token.  Error was : @error (@errorType), @errorDescription.',
          [
            '@error'            => $error->getError(),
            '@errorType'        => $error->getErrorType(),
            '@errorDescription' => $error->getErrorDescription(),
          ]
        );
      }
      $this->logger->error($message);
    }

    return $token;
  }

  private function generateUids(string $cacheKey): array {
    $cache = $this->getCache($cacheKey);
    $uids = [];

    foreach ($cache as $value) {

      $sunetId = $value['sunetId'] ?? NULL;

      if ((NULL === $sunetId) && str_contains($value['email'] ?? '', '@stanford.edu'))
      {
        $this->logger->debug($this->t('Checking email address for SUNet ID of @displayName.',
          ['@displayName' => $value['displayName']]));
          $parts = explode('@', $value['email'] ?? '');
          $sunetId = $parts[0] ?? NULL;

      }

      if (NULL !== $sunetId) {
        $this->logger->debug($this->t('Found sunetid for @displayName.',
          ['@displayName' => $value['displayName']]));
        $uids[] = $sunetId;
      }
    }

    return $uids;
  }

  /**
   * @param string $cid
   *
   * @return array
   */
  public function getCache(string $cid): array {

    if (FALSE === OrglistUtils::checkEnabledByCid($cid)) {
      return [];
    }

    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    if (NULL === $endpoint = $this->splitCid($cid)) {
      return $this->cache->get($cid, TRUE)->data;
    }

    $response = $this->request->request($endpoint);

    if (!in_array($response->getStatusCode(), [200, 201], TRUE)) {

      $message = t('Response : (@code) @reasonPhrase, @base_uri.', [
        '@code'         => $response->getStatusCode(),
        '@reasonPhrase' => $response->getReasonPhrase(),
      ]);

      $this->logger->error($message);

      return $this->cache->get($cid, TRUE)->data ?? [];
    }
    if (NULL === $contents = $response?->getBody()?->getContents()) {
      return $this->cache->get($cid, TRUE)->data ?? [];
    }

    try {
      $results = \GuzzleHttp\json_decode($contents, TRUE);
    } catch (GuzzleException $e) {
      $this->logger->error($e->getMessage(), [
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'code' => $e->getCode(),
      ]);

      if (OrglistUtils::isAdmin()) {
        Drupal::messenger()->addError($e->getMessage());
      }

      return $this->cache->get($cid, TRUE)->data ?? [];
    }

    $results = $this->sortResults($results);

    $expires = (Drupal::config('orglist.settings')
          ->get('cache_lifetime') ?? 60) * 60;

    $this->setCache($cid, $results, $expires);

    return $results;
  }

  /**
   * @param string $cid
   * @param mixed  $data
   * @param int    $expires
   *
   * @return void
   */
  public function setCache(string $cid, mixed $data, int $expires): void {
    $this->cache->set($cid, $data, $expires);
  }

  /**
   * @param string $cid
   *  The cache ID to split
   *
   * @return string
   */
  private function splitCid(string $cid): string {
    if (str_contains($cid, ':')) {
      $parts = explode(':', $cid);
      $endpoint = $parts[1] ?? NULL;
    }

    return $endpoint ?? $cid;

  }

  /**
   * @param array $results
   *
   * @return array
   */
  protected function sortResults(array $results): array {
    foreach ($results as $key => $value) {

      if (array_key_exists('psName', $value)) {
        if (NULL === ($value['psName'] ?? NULL)) {
          $value['psName'] = sprintf('%s-%s', $value['firstName'],
            $value['lastName']);
        }
        $value['psName'] = str_replace(' ', '-',
          mb_strtolower($value['psName']));
      }

      $results[$value['id']] = $value;
      unset($results[$key]);
    }

    return $results;
  }

  /**
   * @return mixed
   */
  public function deleteAllImageCache(): mixed {
    return $this->imageCache->deleteAll();
  }

  /**
   * @param string $type
   *
   * @return mixed
   */
  public function deleteImageCacheOfType(
    string $type = self::CACHE_KEY_IMAGE_FACULTY
  ): mixed {
    return $this->imageCache->delete($type);
  }

  /**
   * @param string $type
   *
   * @return array
   */
  public function rebuildImageCacheOfType(
    string $type = self::CACHE_KEY_IMAGE_FACULTY
  ): array {
    $this->imageCache->delete($type);
    return $this->getImageCache($type);
  }

  /**
   * @param $cid
   *
   * @return mixed
   */
  public function delete($cid): mixed {
    return $this->cache->delete($cid);
  }

  /**
   * @return mixed
   */
  public function deleteAll(): mixed {
    return $this->cache->deleteAll();
  }

  /**
   * @param string $cid
   *
   * @return array
   */
  public function refresh(string $cid): array {
    $this->cache->delete($cid);
    return $this->getCache($cid);
  }

  /**
   * @return array
   */
  public function refreshAll(): array {
    $this->cache->deleteAll();
    return $this->getAll();
  }

  /**
   * @return array
   */
  public function getAll(): array {
    if ($cache = $this->cache->get('people')) {
      return $cache->data;
    }
    /** @var Response[] $responses */
    $responses = $this->request->requestAll();

    $results = [];


    foreach ($responses as $key => $response) {
      try {
        $results[$key] = \GuzzleHttp\json_decode($response['value']->getBody()
          ->getContents(), TRUE);
      } catch (Exception $e) {
        $this->logger->critical($this->t('Could not decode response of @type.',
          ['@type' => $key]));
        $this->logger->critical($e->getMessage());
        return $this->cache->get('people', TRUE)->data;
      }
    }
    unset($key);

    $this->logger->info($this->t('Setting new people cache.'));
    $expires = (int) (Drupal::config('orglist.settings')
          ->get('cache_lifetime') ?? 60) * 60;

    foreach ($results as $key => $value) {
      $value = $this->sortResults($value);

      $this->setCache(sprintf('people:%s', $key), $value,
        (Drupal::time()->getRequestTime() + $expires));
    }

    $this->setCache('people', $results,
      (Drupal::time()->getRequestTime() + $expires));

    return $results;
  }

}
