<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\SamlCert;

use JetBrains\PhpStorm\ArrayShape;
use ReflectionClass;
use Symfony\Component\OptionsResolver\OptionsResolver;

class DistinguishedName
{
    private string $countryName;
    private string $stateOrProvinceName;
    private string $localityName;
    private string $organizationName;
    private string $organizationalUnitName;
    private string $commonName;
    private ?string $emailAddress = null;

    public function __construct(
        #[ArrayShape([
            'countryName' => 'string',
            'stateOrProvinceName' => 'string',
            'localityName' => 'string',
            'organizationName' => 'string',
            'organizationalUnitName' => 'string',
            'commonName' => 'string',
            'emailAddress' => 'string',
    ])]
        array $options = []
    ) {
        $resolver = new OptionsResolver();
        $resolver->setDefined([
           'countryName',
           'stateOrProvinceName',
           'localityName',
           'organizationName',
           'organizationalUnitName',
           'commonName',
           'emailAddress',
        ]);
        $resolver->setRequired([
            'countryName',
            'stateOrProvinceName',
            'localityName',
            'organizationName',
            'organizationalUnitName',
            'commonName',
        ]);

        $resolver->setDefaults([
            'countryName' => 'US',
            'stateOrProvinceName' => 'California',
            'localityName' => 'Stanford',
            'organizationName' => 'Stanford University',
        ]);

        foreach ($resolver->getDefinedOptions() as $optionName) {
            $resolver->setAllowedTypes($optionName, 'string');
        }

        $filteredOptions = $resolver->resolve($options);

        $this->setInitialOptions($filteredOptions);
    }

    public function toArray(): array
    {
        $result = [];

        $reflectionObject = new ReflectionClass($this);
        foreach ($reflectionObject->getProperties() as $key => $value) {
            if (null !== $propertyValue = $value->getValue($this)) {
                $result[$value->getName()] = $propertyValue;
            }
        }

        return $result;
    }
    /**
     * @return string
     */
    public function getStateOrProvinceName(): string
    {
        return $this->stateOrProvinceName;
    }

    /**
     * @param string $stateOrProvinceName
     *
     * @return DistinguishedName
     */
    public function setStateOrProvinceName(string $stateOrProvinceName): DistinguishedName
    {
        $this->stateOrProvinceName = $stateOrProvinceName;
        return $this;
    }

    /**
     * @return string
     */
    public function getLocalityName(): string
    {
        return $this->localityName;
    }

    /**
     * @param string $localityName
     *
     * @return DistinguishedName
     */
    public function setLocalityName(string $localityName): DistinguishedName
    {
        $this->localityName = $localityName;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrganizationName(): string
    {
        return $this->organizationName;
    }

    /**
     * @param string $organizationName
     *
     * @return DistinguishedName
     */
    public function setOrganizationName(string $organizationName): DistinguishedName
    {
        $this->organizationName = $organizationName;
        return $this;
    }

    /**
     * @return string
     */
    public function getOrganizationalUnitName(): string
    {
        return $this->organizationalUnitName;
    }

    /**
     * @param string $organizationalUnitName
     *
     * @return DistinguishedName
     */
    public function setOrganizationalUnitName(string $organizationalUnitName): DistinguishedName
    {
        $this->organizationalUnitName = $organizationalUnitName;
        return $this;
    }

    /**
     * @return string
     */
    public function getCommonName(): string
    {
        return $this->commonName;
    }

    /**
     * @param string $commonName
     *
     * @return DistinguishedName
     */
    public function setCommonName(string $commonName): DistinguishedName
    {
        $this->commonName = $commonName;
        return $this;
    }

    /**
     * @return string
     */
    public function getEmailAddress(): ?string
    {
        return $this->emailAddress;
    }

    /**
     * @param string $emailAddress
     *
     * @return DistinguishedName
     */
    public function setEmailAddress(?string $emailAddress): DistinguishedName
    {
        $this->emailAddress = $emailAddress;
        return $this;
    }
    /**
     * @return string
     */
    public function getCountryName(): string
    {
        return $this->countryName;
    }

    /**
     * @param string $countryName
     *
     * @return DistinguishedName
     */
    public function setCountryName(string $countryName): DistinguishedName
    {
        $this->countryName = $countryName;
        return $this;
    }

    private function setInitialOptions(array $filteredOptions): void
    {
        $this
            ->setCountryName($filteredOptions['countryName'])
            ->setStateOrProvinceName($filteredOptions['stateOrProvinceName'])
            ->setLocalityName($filteredOptions['localityName'])
            ->setOrganizationName($filteredOptions['organizationName'])
            ->setOrganizationalUnitName($filteredOptions['organizationalUnitName'])
            ->setCommonName($filteredOptions['commonName']);
        if (isset($filteredOptions['emailAddress'])) {
            $this->setEmailAddress($filteredOptions['emailAddress'] ?? '');
        }
    }
}
