<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\SamlCert\Tests;

use const OPENSSL_KEYTYPE_RSA;
use PHPUnit\Framework\TestCase;
use Suee\SamlCert\Certificate;
use Suee\SamlCert\DistinguishedName;
use function unlink;

class SignedCertificateTest extends TestCase
{
    public function tearDown(): void
    {
        unlink(__DIR__.'/test.csr');
        unlink(__DIR__.'/test.key');
        unlink(__DIR__.'/test.crt');
    }
    public function testSignedCertificate(): void
    {
        $distinguishedName = new DistinguishedName([
            'commonName' => 'ee.stanford.edu',
            'organizationalUnitName' => 'Electrical Engineering',
        ]);

        $cert = new Certificate($distinguishedName, 4096, OPENSSL_KEYTYPE_RSA, (365 * 10));
        $signed = $cert->export();
        $this->assertStringContainsString('-----BEGIN CERTIFICATE REQUEST-----', $signed->getSigningRequest());
        $this->assertStringContainsString('-----BEGIN PRIVATE KEY-----', $signed->getPrivateKey());
        $this->assertStringContainsString('-----BEGIN CERTIFICATE-----', $signed->getCertificate());
    }

    public function testWriteCertificates(): void
    {
        $distinguishedName = new DistinguishedName([
            'commonName' => 'ee.stanford.edu',
            'organizationalUnitName' => 'Electrical Engineering',
        ]);

        $cert = new Certificate($distinguishedName, 4096, OPENSSL_KEYTYPE_RSA, (365 * 10));
        $signed = $cert->export();
        $signed->writeSigningRequest(__DIR__.'/test.csr');
        $signed->writePrivateKey(__DIR__.'/test.key');
        $signed->writeCertificate(__DIR__.'/test.crt');
        $this->assertStringEqualsFile(__DIR__.'/test.csr', $signed->getSigningRequest());
        $this->assertStringEqualsFile(__DIR__.'/test.key', $signed->getPrivateKey());
        $this->assertStringEqualsFile(__DIR__.'/test.crt', $signed->getCertificate());
    }
}
