<?php declare(strict_types=1);

/*
 * Copyright (c) 2020  https://ee.stanford.edu action@ee.stanford.edu
 *
 * This Academic Free License (the "License") applies to any original work
 * of authorship (the "Original Work") whose owner (the "Licensor") has
 * placed the following licensing notice adjacent to the copyright notice
 * for the Original Work:
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\SamlCert;

use function openssl_error_string;
use const OPENSSL_KEYTYPE_EC;
use const OPENSSL_KEYTYPE_RSA;
use function openssl_pkey_new;
use OpenSSLAsymmetricKey;
use OpenSSLCertificate;
use OpenSSLCertificateSigningRequest;

class Certificate
{
    private OpenSSLCertificateSigningRequest $signingRequest;
    private OpenSSLAsymmetricKey $privateKey;
    private OpenSSLCertificate $certificate;

    public function __construct(
        private readonly DistinguishedName $distinguishedName,
        private readonly int $bits,
        private readonly int $keyType,
        private readonly int $days
    ) {
        if ($this->keyType < OPENSSL_KEYTYPE_RSA || $this->keyType > OPENSSL_KEYTYPE_EC) {
            throw new \RuntimeException('keyType must be one of the OPENSSL_KEYTYPE_* constants.');
        }

        $this->createPrivateKey();
        $this->createCsr();
        $this->createX509();
    }

    /**
     * @return \OpenSSLCertificateSigningRequest
     */
    public function getSigningRequest(): OpenSSLCertificateSigningRequest
    {
        return $this->signingRequest;
    }

    /**
     * @return \OpenSSLAsymmetricKey
     */
    public function getPrivateKey(): OpenSSLAsymmetricKey
    {
        return $this->privateKey;
    }

    /**
     * @return \OpenSSLCertificate
     */
    public function getCertificate(): OpenSSLCertificate
    {
        return $this->certificate;
    }

    /**
     * @return \Suee\SamlCert\DistinguishedName
     */
    public function getDistinguishedName(): DistinguishedName
    {
        return $this->distinguishedName;
    }

    /**
     * @return int
     */
    public function getBits(): int
    {
        return $this->bits;
    }

    /**
     * @return int
     */
    public function getKeyType(): int
    {
        return $this->keyType;
    }

    /**
     * @return int
     */
    public function getDays(): int
    {
        return $this->days;
    }

    public function export(): SignedCertificate
    {
        openssl_csr_export($this->signingRequest, $signingRequest);
        openssl_x509_export($this->certificate, $certificate);
        openssl_pkey_export($this->privateKey, $privateKey);

        return new SignedCertificate($signingRequest, $privateKey, $certificate);
    }

    private function createPrivateKey(): void
    {
        $privateKey = openssl_pkey_new([
            'private_key_bits' => $this->bits,
            'private_key_type' => $this->keyType,
        ]);

        if (false === $privateKey) {
            throw new \RuntimeException(openssl_error_string() ?: 'An Error Occurred');
        }

        $this->privateKey = $privateKey;
    }
    private function createCsr(): void
    {
        if (false === $signingRequest = openssl_csr_new($this->distinguishedName->toArray(), $this->privateKey, ['digest_alg' => 'sha256'])) {
            throw new \RuntimeException(openssl_error_string());
        }

        $this->signingRequest = $signingRequest;
    }

    private function createX509(): void
    {
        if (false === $x509 = openssl_csr_sign($this->signingRequest, null, $this->privateKey, $this->days, ['digest_alg' => 'sha256'])) {
            throw new \RuntimeException(openssl_error_string());
        }

        $this->certificate = $x509;
    }
}
