<?php declare(strict_types=1);

/*
 * Copyright 2025 Stanford Electrical Engineering
 * https:/ee.stanford.edu <reric@ee.stanford.edu>
 *
 * Licensed under the Academic Free License version 3.0
 *
 */

namespace Suee\StanfordAuthorization\Authorization;

use function base64_encode;
use function error_log;
use JsonException;
use Psr\Http\Client\ClientExceptionInterface;
use Psr\Http\Client\ClientInterface;
use Psr\SimpleCache\CacheInterface;
use PsrDiscovery\Discover;
use PsrDiscovery\Exceptions\SupportPackageNotFoundException;

use SensitiveParameter;
use function sprintf;
use Suee\StanfordAuthorization\Authorization\Factory\PsrFactory;

/**
 * Class Token
 *
 * @package Suee\CAP\Authorization
 */
class TokenFactory
{
    /**
     *
     */
    public const string DEFAULT_TOKEN_URL = 'https://authz.stanford.edu/oauth/token';

    private ?ClientInterface $client;

    public function __construct(
        private readonly string $username,
        #[SensitiveParameter]
        private readonly string $password,
        ?ClientInterface $client = null,
        private ?CacheInterface $cache = null,
    ) {
        if (null === $client) {
            $this->client = PsrFactory::createClient();
        } else {
            $this->client = $client;
        }
    }

    public function getToken(): ?Token
    {
        $cache = $this->getCacheDriver();

        if ($cache?->has('tokenStorage')) {
            return $cache?->get('tokenStorage');
        }

        if (null === $response = $this->requestNewToken()) {
            return null;
        }

        try {
            $jsonArray = json_decode($response, true, flags: JSON_THROW_ON_ERROR);
        } catch (JsonException $e) {
            error_log(sprintf('There was an error decoding the returned json on line %s of %s.', $e->getLine(), $e->getFile()), 0);
            return null;
        }

        $token = new Token($jsonArray);

        $cache?->set('tokenStorage', $token, $token->getExpiresIn() - 300);

        return $token;
    }

    private function getCacheDriver(): ?CacheInterface
    {
        if ($this->cache === null) {
            try {
                $this->cache = Discover::cache(true);
            } catch (SupportPackageNotFoundException $e) {
                error_log($e->getMessage());
            }
        }

        return $this->cache;
    }

    private function createBasicAuth(): string
    {
        return sprintf(
            'Basic %s',
            base64_encode(
                sprintf('%s:%s', $this->username, $this->password)
            )
        );
    }
    private function requestNewToken(): ?string
    {
        $uri = PsrFactory::createPsr17Factory()
            ->createUri(self::DEFAULT_TOKEN_URL)
            ->withQuery(http_build_query(['grant_type' => 'client_credentials']))
            ->withUserInfo($this->username, $this->password)
        ;

        $request = PsrFactory::createPsr17Factory()
            ->createRequest('POST', $uri)
        ;

        try {
            $response = $this->client->sendRequest($request);
        } catch (ClientExceptionInterface $e) {
            error_log($e->getMessage());
            return null;
        }

        if ($response->getStatusCode() !== 200) {
            error_log(sprintf('Error requesting token in %s on line %d, with code %d and message %s.', __FILE__, __LINE__, $response->getStatusCode(), $response->getReasonPhrase()));
            return null;
        }

        return $response->getBody()->getContents();
    }
}
