<?php declare(strict_types=1);

namespace Tests\Suee\StanfordAuthorization\Authorization;

use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;
use Psr\SimpleCache\CacheInterface;
use Suee\StanfordAuthorization\Authorization\Token;
use Suee\StanfordAuthorization\Authorization\TokenFactory;

#[CoversClass(TokenFactory::class)]
class TokenFactoryTest extends TestCase
{
    private ClientInterface $mockClient;
    private CacheInterface $mockCache;
    private TokenFactory $factory;

    protected function setUp(): void
    {
        $this->mockClient = $this->createMock(ClientInterface::class);
        $this->mockCache = $this->createMock(CacheInterface::class);
        
        $this->factory = new TokenFactory(
            username: 'test_user',
            password: 'test_password',
            client: $this->mockClient,
            cache: $this->mockCache
        );
    }

    public function testGetTokenUsesCachePriority(): void
    {
        $mockToken = $this->createMock(Token::class);

        $this->mockCache->expects($this->once())
            ->method('has')
            ->with('tokenStorage')
            ->willReturn(true);

        $this->mockCache->expects($this->once())
            ->method('get')
            ->willReturn($mockToken);

        $result = $this->factory->getToken();
        $this->assertSame($mockToken, $result);
    }

    public function testRequestNewTokenCalculatesCorrectTtl(): void
    {
        $this->mockCache->method('has')->willReturn(false);

        $json = json_encode([
            'access_token' => 'abc',
            'token_type' => 'Bearer',
            'expires_in' => 600, // 10 minutes
            'scope' => '',
            'jti' => ''
        ]);

        $response = $this->createMock(ResponseInterface::class);
        $stream = $this->createMock(StreamInterface::class);
        
        $stream->method('getContents')->willReturn($json);
        $response->method('getStatusCode')->willReturn(200);
        $response->method('getBody')->willReturn($stream);

        $this->mockClient->method('sendRequest')->willReturn($response);

        // Logic check: (expires_in - 300) => (600 - 300) = 300
        $this->mockCache->expects($this->once())
            ->method('set')
            ->with('tokenStorage', $this->isInstanceOf(Token::class), 300);

        $this->factory->getToken();
    }
}